import {Type} from "@angular/core";
import {Navigate} from "@ngxs/router-plugin";
import {
  ActionCompletion,
  ActionOptions,
  Actions,
  ensureStoreMetadata,
  ofActionDispatched,
  StateContext,
  Store,
} from "@ngxs/store";
import {
  defer,
  from,
  merge,
  MonoTypeOperatorFunction,
  Observable,
  of,
  pipe,
  zip,
} from "rxjs";
import {
  catchError,
  concatMap,
  filter,
  map,
  mapTo,
  takeUntil,
  tap,
  toArray,
} from "rxjs/operators";
import {RoutesEnum} from "../../enums";
import {SolidifyError} from "../../errors";
import {ErrorHelper} from "../../helpers/error.helper";
import {Lodash} from "../../lib-wrapped/lodash.model";
import {
  BaseResourceType,
  CollectionTyped,
} from "../../models";
import {ValidationErrorDto} from "../../models/errors/error-dto.model";
import {FormError} from "../../models/errors/form-error.model";
import {ModelFormControlEvent} from "../../models/forms/model-form-control-event.model";
import {QueryParameters} from "../../models/query-parameters/query-parameters.model";
import {NotifierService} from "../../models/services/notifier-service.model";
import {BaseResourceState} from "../../models/stores/base-resource.state";
import {BaseStateModel} from "../../models/stores/base-state.model";
import {
  ActionSubActionCompletionsWrapper,
  BaseAction,
  BaseSubAction,
} from "../../models/stores/base.action";
import {StoreActionClass} from "../../models/stores/state-action.model";
import {SubResourceUpdateModel} from "../../models/stores/sub-resource-update.model";
import {AssociationNoSqlReadOnlyStateModel} from "../../stores/association-no-sql-read-only";
import {BasicState} from "../../stores/base";
import {CompositionStateModel} from "../../stores/composition";
import {ResourceStateModel} from "../../stores/resource/resource-state.model";
import {
  isFunction,
  isNonEmptyArray,
  isNonEmptyString,
  isNullOrUndefined,
  isTruthyObject,
  isUndefined,
} from "../../tools/is/is.tool";
import {ObjectUtil} from "../object.util";
import {SOLIDFIY_ERRORS} from "../validations/validation.util";
import {SolidifyMetadataUtil} from "./solidify-metadata.util";

// @dynamic
export class StoreUtil {
  private static readonly ATTRIBUTE_STATE_NAME: string = "name";
  private static readonly NGXS_META_OPTIONS_KEY: string = "NGXS_OPTIONS_META";

  // TODO : Enhance typing ?
  static initState<T>(baseConstructor: Function, constructor: Function, nameSpace: T): void {
    const baseMeta = SolidifyMetadataUtil.ensureStoreSolidifyMetadata<T>(baseConstructor as any);
    const defaultActions = baseMeta.defaultActions;
    if (Array.isArray(defaultActions)) {
      const meta = SolidifyMetadataUtil.ensureStoreSolidifyMetadata<T>(constructor as any);
      const excludedRegisteredDefaultActionFns = meta.excludedRegisteredDefaultActionFns;
      const safeExcludedFns = Array.isArray(excludedRegisteredDefaultActionFns) ? excludedRegisteredDefaultActionFns : [];
      defaultActions.forEach(({fn, callback, options}) => {
        if (safeExcludedFns.indexOf(fn) < 0) {
          this.initDefaultAction(constructor, fn, callback(nameSpace), options);
        }
      });
    }
  }

  static getDefaultData(constructor: Function): ResourceStateModel<BaseResourceType> {
    const meta = ensureStoreMetadata(constructor as any);
    return meta.defaults;
  }

  static initDefaultAction(constructor: Function, fn: string, storeActionClass: StoreActionClass, options?: ActionOptions): void {
    if (storeActionClass === undefined) {
      return;
    }
    const meta = ensureStoreMetadata(constructor as any);
    const type = storeActionClass.type;
    if (!type) {
      throw new Error(`Action ${storeActionClass.name} is missing a static "type" property`);
    }
    if (!meta.actions[type]) {
      meta.actions[type] = [];
    }
    meta.actions[type].push({
      fn,
      options: options || {},
      type,
    });
  }

  static getQueryParametersToApply(queryParameters: QueryParameters, ctx: StateContext<BaseResourceState>): QueryParameters {
    return queryParameters == null ? ctx.getState().queryParameters : queryParameters;
  }

  static updateQueryParameters<T>(ctx: StateContext<ResourceStateModel<T>> | StateContext<CompositionStateModel<T>> | StateContext<AssociationNoSqlReadOnlyStateModel<T>>, list: CollectionTyped<T> | null | undefined): QueryParameters {
    const queryParameters = ObjectUtil.clone(ctx.getState().queryParameters);
    const paging = ObjectUtil.clone(queryParameters.paging);
    paging.length = list === null || list === undefined ? 0 : list._page.totalItems;
    queryParameters.paging = paging;
    return queryParameters;
  }

  static dispatchParallelActionAndWaitForSubActionsCompletion<T>(ctx: StateContext<T> | Store, actionSubActionCompletionsWrappers: ActionSubActionCompletionsWrapper[]): Observable<boolean> {
    if (actionSubActionCompletionsWrappers.length === 0) {
      return of(true);
    }

    const actions = new Array(actionSubActionCompletionsWrappers.length);
    const subActionCompletionObservables = [];
    actionSubActionCompletionsWrappers.forEach(
      (actionSubActionCompletionsWrapper, i) => {
        actions[i] = actionSubActionCompletionsWrapper.action;
        if (isNonEmptyArray(actionSubActionCompletionsWrapper.subActionCompletions)) {
          subActionCompletionObservables.push(
            merge(...actionSubActionCompletionsWrapper.subActionCompletions)
              .pipe(
                filter(actionCompletion => this.hasParentAction(actionCompletion.action, actionSubActionCompletionsWrapper.action)),
              ),
          );
        }
      },
    );
    if (subActionCompletionObservables.length === 0) {
      return ctx.dispatch(actions).pipe(mapTo(true));
    }
    return zip(
      zip(...subActionCompletionObservables),
      defer(() => ctx.dispatch(actions)),
    ).pipe(
      map(values => (values[0] as ActionCompletion<BaseSubAction<BaseAction>, Error>[]).every(actionCompletion => actionCompletion.result.successful)),
    );
  }

  static dispatchSequentialActionAndWaitForSubActionsCompletion<T>(ctx: StateContext<T> | Store, actionSubActionCompletionsWrappers: ActionSubActionCompletionsWrapper[]): Observable<boolean> {
    const obs = from(actionSubActionCompletionsWrappers)
      .pipe(
        concatMap((action) => {
          let subActionCompletionObs = of(undefined);

          if (isNonEmptyArray(action.subActionCompletions)) {
            subActionCompletionObs = merge(...action.subActionCompletions)
              .pipe(
                filter(actionCompletion => this.hasParentAction(actionCompletion.action, action.action)),
              );
          }

          return zip(
            zip(subActionCompletionObs),
            defer(() => ctx.dispatch(action.action)),
          ).pipe(
            map(values => {
              if (values[0].length === 1 && isUndefined(values[0][0])) {
                // console.debug("There is no subaction to wait for this action", values);
                return true;
              } else {
                const success = (values[0] as ActionCompletion<BaseSubAction<BaseAction>, Error>[]).every(actionCompletion => actionCompletion.result.successful);
                // console.debug("There is subaction to wait for this action. Success ? => " + success, values);
                return success;
              }
            }),
          );
        }),
      );

    return obs.pipe(
      toArray(),
      map((value) => {
        const allSuccess = value.every(v => v === true);
        // console.debug("All value success ? => " + allSuccess, value);
        return allSuccess;
      }));
  }

  static hasParentAction(action: BaseAction, parentAction: BaseAction): boolean {
    if (!isTruthyObject(action) || !isTruthyObject(parentAction)) {
      return false;
    }
    const parents: BaseAction[] = [];
    let currentAction = action;
    while (isTruthyObject(currentAction.parentAction)) {
      const currentParentAction = currentAction.parentAction;
      if (currentParentAction === parentAction) {
        return true;
      }
      if (currentParentAction === action || parents.includes(currentParentAction)) {
        return false;
      }
      parents.push(currentParentAction);
      currentAction = currentParentAction;
    }
    return false;
  }

  static isLoadingState(baseState: BaseStateModel): boolean {
    return baseState.isLoadingCounter > 0;
  }

  static cancelUncompleted<T>(ctx: StateContext<BaseStateModel>, actions$: Actions, allowedTypes: any[], noLoadingCounterDecrement?: boolean): MonoTypeOperatorFunction<T> {
    return pipe(
      takeUntil(
        actions$.pipe(
          ofActionDispatched(...allowedTypes),
          tap(() => {
              if (!noLoadingCounterDecrement) {
                ctx.patchState({
                  isLoadingCounter: ctx.getState().isLoadingCounter - 1,
                });
              }
            },
          ),
        ),
      ),
    );
  }

  static determineSubResourceChange(oldList: string[], newList: string[]): SubResourceUpdateModel {
    const subResourceUpdate: SubResourceUpdateModel = new SubResourceUpdateModel();
    const diff: string[] = Lodash.xor(oldList, newList);
    diff.forEach(d => {
      if (Lodash.includes(oldList, d)) {
        subResourceUpdate.resourceToRemoved.push(d);
      } else {
        subResourceUpdate.resourceToAdd.push(d);
      }
    });
    return subResourceUpdate;
  }

  static catchValidationErrors<T>(ctx: StateContext<ResourceStateModel<T>>, modelFormControlEvent: ModelFormControlEvent<T>): MonoTypeOperatorFunction<T> {
    return catchError(error => {
      let errorToTreat = error;
      if (error instanceof SolidifyError) {
        errorToTreat = error.nativeError;
      }
      const validationErrors = ErrorHelper.extractValidationErrors(errorToTreat);
      if (isNonEmptyArray(validationErrors)) {
        if (isTruthyObject(modelFormControlEvent.formControl)) {
          this.applyValidationErrorsOnFormControl(modelFormControlEvent, validationErrors);
        }
      }
      throw error;
    });
  }

  private static applyValidationErrorsOnFormControl<T>(modelFormControlEvent: ModelFormControlEvent<T>, validationErrors: ValidationErrorDto[]): void {
    Object.keys(modelFormControlEvent.formControl.value).forEach(fieldName => {
      const formControl = modelFormControlEvent.formControl.get(fieldName);
      if (!isTruthyObject(formControl)) {
        return;
      }
      const hasToUpdateValueAndValidity = (isTruthyObject(formControl[SOLIDFIY_ERRORS]) && formControl[SOLIDFIY_ERRORS].errorsFromBackend !== null) || !isNullOrUndefined(formControl.errors);
      delete formControl[SOLIDFIY_ERRORS];
      if (hasToUpdateValueAndValidity) {
        formControl.updateValueAndValidity();
      }
    });
    validationErrors.forEach(error => {
      const formControl = modelFormControlEvent.formControl.get(error.fieldName);
      let errors = formControl.errors;
      if (isNullOrUndefined(errors)) {
        errors = {};
      }
      formControl[SOLIDFIY_ERRORS] = {errorsFromBackend: error.errorMessages} as FormError;
      Object.assign(errors, {errorsFromBackend: error.errorMessages});
      formControl.setErrors(errors);
      formControl.markAsTouched();
      // formControl.updateValueAndValidity(); ==> Remove error set with formControl.setErrors(errors);...
    });
    modelFormControlEvent.changeDetectorRef.detectChanges();
  }

  static notifySuccess(notifierService: NotifierService, textToTranslate: string, messageParam: Object | undefined = undefined): boolean {
    if (isTruthyObject(notifierService)) {
      if (isNonEmptyString(textToTranslate)) {
        notifierService.showSuccess(textToTranslate, messageParam);
        return true;
      }
    }
    return false;
  }

  static notifyError(notifierService: NotifierService, textToTranslate: string, messageParam: Object | undefined = undefined): boolean {
    if (isTruthyObject(notifierService)) {
      if (isNonEmptyString(textToTranslate)) {
        notifierService.showError(textToTranslate, messageParam);
        return true;
      }
    }
    return false;
  }

  static navigateIfDefined(ctx: StateContext<any>, route: RoutesEnum | undefined | ((resId: string) => string), resId: string | undefined): void {
    if (!isNullOrUndefined(route)) { // TODO MANAGE CASE DELETE WITHOUT RESID
      if (isFunction(route)) {
        route = route(resId);
      }
      ctx.dispatch(new Navigate([route]));
    }
  }

  static getStateNameFromInstance<TStateModel extends BaseStateModel = BaseStateModel>(store: BasicState<TStateModel>): string {
    return this.getStateNameFromClass(store.constructor as Type<BasicState<BaseStateModel>>);
  }

  static getStateNameFromClass<TStateModel extends BaseStateModel = BaseStateModel>(ctor: Type<BasicState<TStateModel>>): string {
    return this.getStateFromClass(ctor)[this.ATTRIBUTE_STATE_NAME];
  }

  static getStateFromInstance<T = any, TStateModel extends BaseStateModel = BaseStateModel>(store: BasicState<TStateModel>): T {
    return this.getStateFromClass(store.constructor as Type<BasicState<BaseStateModel>>);
  }

  static getStateFromClass<T = any, TStateModel extends BaseStateModel = BaseStateModel>(ctor: Type<BasicState<TStateModel>>): T {
    return ctor[this.NGXS_META_OPTIONS_KEY];
  }
}
