import {ensureStoreMetadata} from "@ngxs/store";
import {RegisteredDefaultAction} from "../../decorators";
import {
  BaseAction,
  StoreActionClass,
} from "../../models";
import {BaseState} from "../../stores/base/base.state";
import {ResourceNameSpace} from "../../stores/resource";
import {StubActionBis} from "../../stubs/stub-action";
import {StubEmptyClass} from "../../stubs/stub-empty-class";
import {
  SOLIDIFY_META_KEY,
  SolidifyMetadataUtil,
} from "./solidify-metadata.util";
import {StoreUtil} from "./store.util";

describe("StoreUtil", () => {
  describe("initState", () => {
    it("should register 0 action when no action in solidify meta", () => {
      const baseConstructor = {};
      const childConstructor = {};

      const functionName = "GetAll";
      const action = new StubActionBis();
      action.type = "[MyState] Get All";

      const spy = spyOn(StoreUtil, "initDefaultAction");
      StoreUtil.initState(baseConstructor.constructor, childConstructor.constructor, action as StoreActionClass | any);
      expect(spy).toHaveBeenCalledTimes(0);
      baseConstructor.constructor[SOLIDIFY_META_KEY] = undefined;
      childConstructor.constructor[SOLIDIFY_META_KEY] = undefined;
    });

    it("should register 1 action when 1 action in solidify meta of base class", () => {
      const baseConstructor = {};
      const childConstructor = {};
      const functionName = "GetAll";
      const action = new StubActionBis();
      action.type = "[MyState] Get All";

      const baseMeta = SolidifyMetadataUtil.ensureStoreSolidifyMetadata<ResourceNameSpace>(baseConstructor.constructor as any);
      baseMeta.defaultActions = [{
        fn: functionName,
        callback: () => null,
        options: {},
      }];
      const spy = spyOn(StoreUtil, "initDefaultAction");
      StoreUtil.initState(baseConstructor.constructor, childConstructor.constructor, action as StoreActionClass | any);
      expect(spy).toHaveBeenCalledTimes(1);
      baseConstructor.constructor[SOLIDIFY_META_KEY] = undefined;
      childConstructor.constructor[SOLIDIFY_META_KEY] = undefined;
    });

    it("should register 2 action when 2 action in solidify meta of base class", () => {
      const baseConstructor = {};
      const childConstructor = {};
      const functionName = "GetAll";
      const action = new StubActionBis();
      action.type = "[MyState] Get All";

      const baseMeta = SolidifyMetadataUtil.ensureStoreSolidifyMetadata<ResourceNameSpace>(baseConstructor.constructor as any);
      baseMeta.defaultActions = [
        {
          fn: functionName + 1,
          callback: () => null,
          options: {},
        },
        {
          fn: functionName + 2,
          callback: () => null,
          options: {},
        },
      ];
      const spy = spyOn(StoreUtil, "initDefaultAction");
      StoreUtil.initState(baseConstructor.constructor, childConstructor.constructor, action as StoreActionClass | any);
      expect(spy).toHaveBeenCalledTimes(2);
      baseConstructor.constructor[SOLIDIFY_META_KEY] = undefined;
      childConstructor.constructor[SOLIDIFY_META_KEY] = undefined;
    });

    it("should register 1 action when 2 action in solidify meta of base class and 1 excluded in state class", () => {
      const baseConstructor = {};
      const childConstructor = {};
      const functionName = "GetAll";
      const action = new StubActionBis();
      action.type = "[MyState] Get All";

      const baseMeta = SolidifyMetadataUtil.ensureStoreSolidifyMetadata<ResourceNameSpace>(baseConstructor.constructor as any);
      baseMeta.defaultActions = [
        {
          fn: functionName + 1,
          callback: () => null,
          options: {},
        },
        {
          fn: functionName + 2,
          callback: () => null,
          options: {},
        },
      ];

      const meta = SolidifyMetadataUtil.ensureStoreSolidifyMetadata<ResourceNameSpace>(childConstructor.constructor as any);
      meta.excludedRegisteredDefaultActionFns = [
        functionName + 2,
      ];

      const spy = spyOn(StoreUtil, "initDefaultAction");
      StoreUtil.initState(baseConstructor.constructor, childConstructor.constructor, action as StoreActionClass | any);
      expect(spy).toHaveBeenCalledTimes(1);
      baseConstructor.constructor[SOLIDIFY_META_KEY] = undefined;
      childConstructor.constructor[SOLIDIFY_META_KEY] = undefined;
    });
  });

  describe("initDefaultAction", () => {
    it("should register action on NGXS metadata", () => {
      const constructor = {};
      const functionName = "GetAll";
      const action = new StubActionBis();
      action.type = "[MyState] Get All";
      const metaNgxs = ensureStoreMetadata(constructor.constructor as any);
      expect(metaNgxs).not.toBeNull();
      expect(metaNgxs).not.toBeUndefined();
      let actionsRegisteredInNgXs = Object.getOwnPropertyNames(metaNgxs.actions);
      expect(actionsRegisteredInNgXs.length).toBe(0);

      StoreUtil.initDefaultAction(constructor.constructor, functionName, action as StoreActionClass | any);

      actionsRegisteredInNgXs = Object.getOwnPropertyNames(metaNgxs.actions);
      expect(actionsRegisteredInNgXs.length).toBe(1);
      expect(actionsRegisteredInNgXs[0]).toBe(action.type);
      const actionInNgxs: RegisteredDefaultAction<ResourceNameSpace> | any = metaNgxs.actions[actionsRegisteredInNgXs[0]];
      expect(actionInNgxs.length).toBe(1);
      expect(actionInNgxs[0].fn).toBe(functionName);
      expect(Object.getOwnPropertyNames(actionInNgxs[0].options).length).toBe(0);
      expect(actionInNgxs[0].type).toBe(action.type);

      constructor.constructor[SOLIDIFY_META_KEY] = undefined;
    });
  });

  describe("hasParentAction", () => {
    it("should return false when it is not parent", () => {
      const parentAction = {} as BaseAction;
      const action = {} as BaseAction;
      expect(StoreUtil.hasParentAction(action, parentAction)).toBeFalsy();
    });

    it("should return true when it is parent directly", () => {
      const parentAction = {} as BaseAction;
      const action = {parentAction: parentAction} as BaseAction;
      expect(StoreUtil.hasParentAction(action, parentAction)).toBeTruthy();
    });

    it("should return true when it is parent indirectly directly", () => {
      const parentAction = {} as BaseAction;
      const intermediateAction = {parentAction: parentAction} as BaseAction;
      const action = {parentAction: intermediateAction} as BaseAction;
      expect(StoreUtil.hasParentAction(action, parentAction)).toBeTruthy();
    });
  });

  describe("isLoadingState", () => {
    it("should return true when isLoadingCounter > 0", () => {
      expect(StoreUtil.isLoadingState({isLoadingCounter: 1})).toBeTruthy();
    });

    it("should return true when isLoadingCounter = 0", () => {
      expect(StoreUtil.isLoadingState({isLoadingCounter: 0})).toBeFalsy();
    });

    it("should return true when isLoadingCounter = -1", () => {
      expect(StoreUtil.isLoadingState({isLoadingCounter: -1})).toBeFalsy();
    });
  });

  describe("determineSubResourceChange", () => {
    it("should return (0 add / 0 remove) when list empty", () => {
      const oldList = [];
      const newList = [];
      const result = StoreUtil.determineSubResourceChange(oldList, newList);
      expect(result.resourceToRemoved.length).toBe(0);
      expect(result.resourceToAdd.length).toBe(0);
    });

    it("should return (1 add / 0 remove) when list (1 new / 0 old)", () => {
      const oldList = [];
      const newList = ["1"];
      const result = StoreUtil.determineSubResourceChange(oldList, newList);
      expect(result.resourceToRemoved.length).toBe(0);
      expect(result.resourceToAdd.length).toBe(1);
    });

    it("should return (0 add / 1 remove) when list (0 new / 1 old)", () => {
      const oldList = ["1"];
      const newList = [];
      const result = StoreUtil.determineSubResourceChange(oldList, newList);
      expect(result.resourceToRemoved.length).toBe(1);
      expect(result.resourceToAdd.length).toBe(0);
    });

    it("should return (0 add / 0 remove) when list (1 new / 1 old) same", () => {
      const oldList = ["1"];
      const newList = ["1"];
      const result = StoreUtil.determineSubResourceChange(oldList, newList);
      expect(result.resourceToRemoved.length).toBe(0);
      expect(result.resourceToAdd.length).toBe(0);
    });

    it("should return (1 add / 1 remove) when list (2 new / 2 old) with one element same", () => {
      const oldList = ["1", "2"];
      const newList = ["1", "3"];
      const result = StoreUtil.determineSubResourceChange(oldList, newList);
      expect(result.resourceToRemoved.length).toBe(1);
      expect(result.resourceToAdd.length).toBe(1);
    });

    it("should return (1 add / 1 remove) when list (1 new / 1 old) with different element", () => {
      const oldList = ["2"];
      const newList = ["3"];
      const result = StoreUtil.determineSubResourceChange(oldList, newList);
      expect(result.resourceToRemoved.length).toBe(1);
      expect(result.resourceToAdd.length).toBe(1);
    });
  });

  describe("getState", () => {
    it("should return name of the state", () => {
      const state = new StubEmptyClass();
      state.constructor["NGXS_OPTIONS_META"] = {name: "stateName"};
      expect(StoreUtil.getStateNameFromInstance(state as BaseState<any>)).toBe("stateName");
      state.constructor["NGXS_OPTIONS_META"] = undefined;
    });
  });
});
