import {OrderApiEnum} from "../enums/order-api.enum";
import {OrderEnum} from "../enums/order.enum";
import {
  MappingObject,
  Paging,
} from "../models";
import {QueryParameters} from "../models/query-parameters/query-parameters.model";
import {
  isFalse,
  isNullOrUndefined,
} from "../tools/is/is.tool";
import {ObjectUtil} from "./object.util";

export class QueryParametersUtil {
  public static clone(queryParameters: QueryParameters): QueryParameters {
    if (isNullOrUndefined(queryParameters)) {
      return null;
    }
    const newQueryParameters = ObjectUtil.clone(queryParameters);
    if (!isNullOrUndefined(newQueryParameters.paging)) {
      newQueryParameters.paging = ObjectUtil.clone(newQueryParameters.paging);
    }
    if (!isNullOrUndefined(newQueryParameters.sort)) {
      newQueryParameters.sort = ObjectUtil.clone(newQueryParameters.sort);
    }
    if (!isNullOrUndefined(newQueryParameters.search)) {
      newQueryParameters.search = ObjectUtil.clone(newQueryParameters.search);
      if (!isNullOrUndefined(newQueryParameters.search.searchItems)) {
        newQueryParameters.search.searchItems = ObjectUtil.clone(newQueryParameters.search.searchItems);
      }
    }
    return newQueryParameters;
  }

  public static getSortString(queryParameters: QueryParameters): string | null {
    if (isNullOrUndefined(queryParameters) || isNullOrUndefined(queryParameters.sort)) {
      return null;
    }
    return queryParameters.sort.field + "," + (queryParameters.sort.order === OrderEnum.descending ? OrderApiEnum.descending : OrderApiEnum.ascending);
  }

  public static getPageIndex(queryParameters: QueryParameters): number | null {
    if (isNullOrUndefined(queryParameters) || isNullOrUndefined(queryParameters.paging)) {
      return null;
    }
    return queryParameters.paging.pageIndex;
  }

  public static getPageSize(queryParameters: QueryParameters): number | null {
    if (isNullOrUndefined(queryParameters) || isNullOrUndefined(queryParameters.paging)) {
      return null;
    }
    return queryParameters.paging.pageSize;
  }

  public static getSearchItems(queryParameters: QueryParameters): MappingObject<string> | null {
    if (isNullOrUndefined(queryParameters) || isNullOrUndefined(queryParameters.search)) {
      return null;
    }
    return queryParameters.search.searchItems;
  }

  public static resetToFirstPage(queryParameters: QueryParameters, preservePageSize: boolean = true): QueryParameters {
    if (isNullOrUndefined(queryParameters)) {
      return new QueryParameters();
    }
    queryParameters = ObjectUtil.clone(queryParameters);
    if (isNullOrUndefined(queryParameters.paging) || isFalse(preservePageSize)) {
      queryParameters.paging = new Paging();
      return queryParameters;
    }
    const paging = ObjectUtil.clone(queryParameters.paging);
    paging.pageIndex = 0;
    queryParameters.paging = paging;
    return queryParameters;
  }
}
