import {
  isNullOrUndefined,
  isTruthyObject,
  isWhiteString,
} from "../tools";

// @dynamic
export class ObjectUtil {
  static clone<T>(object: T): T {
    // Warning, it's not a deep copy !
    return Object.assign({}, object);
  }

  static getValue(object: object, key: string, defaultValue: any = undefined): any {
    if (isNullOrUndefined(object)) {
      return defaultValue;
    }
    return object[key];
  }

  static getNestedValue(object: object, nestedKey: string, defaultValue: any = undefined): any {
    if (isNullOrUndefined(nestedKey) || isWhiteString(nestedKey) || isNullOrUndefined(object)) {
      return defaultValue;
    }
    nestedKey = nestedKey.replace(/\[(\w+)\]/g, ".$1"); // convert indexes to properties
    nestedKey = nestedKey.replace(/^\./, "");           // strip a leading dot
    const listKey = nestedKey.split(".");
    for (let i = 0, n = listKey.length; i < n; ++i) {
      const key = listKey[i];
      if (key in object) {
        object = object[key];
      } else {
        return defaultValue;
      }
    }
    return object;
  }

  static keys(object: object): string[] {
    if (!isTruthyObject(object)) {
      return [];
    } else {
      return Object.keys(object);
    }
  }

  static values(object: object): any[] {
    if (!isTruthyObject(object)) {
      return [];
    } else {
      return Object.values(object);
    }
  }

  static hasKey(object: object, key: string): boolean {
    const keys = this.keys(object);
    return keys.includes(key);
  }

  static clearKey(object: object, key: string): boolean {
    return this.hasKey(object, key) && delete object[key];
  }

  static clearKeys(object: object): number {
    return this.keys(object).reduce((result, key) => {
      if (delete object[key]) {
        return result + 1;
      } else {
        return result;
      }
    }, 0);
  }
}
