import {MappingObject} from "../models";
import {
  isEquality,
  isFunction,
  isMap,
  isTruthyObject,
  isUndefined,
} from "../tools";
import {ObjectUtil} from "./object.util";

// @dynamic
export class MappingObjectUtil {
  static toMap<T>(mappingObject: MappingObject<T>): Map<string, T> {
    if (isMap(mappingObject)) {
      return mappingObject;
    } else {
      return new Map(Object.entries(mappingObject).filter(([key, value]) => !isUndefined(value)));
    }
  }

  static toObject<T>(mappingObject: MappingObject<T>): { [key: string]: T } {
    if (isMap(mappingObject)) {
      const result = {};
      mappingObject.forEach((value, key) => result[key] = value);
      return result;
    } else {
      return mappingObject;
    }
  }

  static clear<T>(mappingObject: MappingObject<T>): void {
    if (isTruthyObject(mappingObject)) {
      if (isMap(mappingObject)) {
        mappingObject.clear();
      } else {
        ObjectUtil.clearKeys(mappingObject);
      }
    }
  }

  static delete<T>(mappingObject: MappingObject<T>, key: string): boolean {
    if (isTruthyObject(mappingObject)) {
      if (isMap(mappingObject)) {
        return mappingObject.delete(key);
      } else {
        return ObjectUtil.clearKey(mappingObject, key);
      }
    }
    return false;
  }

  static size<T>(mappingObject: MappingObject<T>): number {
    return this.toMap(mappingObject).size;
  }

  static forEach<T>(mappingObject: MappingObject<T>, callback: (value: T, key: string, mappingObject: MappingObject<T>) => void, thisArg?: any): void {
    if (!isFunction(callback)) {
      return;
    }
    if (isTruthyObject(mappingObject)) {
      if (isMap(mappingObject)) {
        mappingObject.forEach(callback, thisArg);
      } else {
        ObjectUtil.keys(mappingObject).forEach(key => {
          callback.call(thisArg, mappingObject[key], key, mappingObject);
        });
      }
    }
  }

  static get<T>(mappingObject: MappingObject<T>, key: string): T | undefined {
    if (isTruthyObject(mappingObject)) {
      if (isMap(mappingObject)) {
        return mappingObject.get(key);
      } else {
        return mappingObject[key];
      }
    }
    return undefined;
  }

  static has<T>(mappingObject: MappingObject<T>, key: string, checkValueIsDefined?: boolean): boolean {
    if (checkValueIsDefined) {
      return !isUndefined(this.get(mappingObject, key));
    }
    if (isTruthyObject(mappingObject)) {
      if (isMap(mappingObject)) {
        return mappingObject.has(key);
      } else {
        return ObjectUtil.hasKey(mappingObject, key);
      }
    }
    return false;
  }

  static set<T>(mappingObject: MappingObject<T>, key: string, value: T): boolean {
    if (isTruthyObject(mappingObject)) {
      const currentValue = this.get(mappingObject, key);
      if (isMap(mappingObject)) {
        mappingObject.set(key, value);
      } else {
        mappingObject[key] = value;
      }
      return !isEquality(currentValue, value);
    }
    return false;
  }
}
