import {MomentDate} from "../lib-wrapped/moment.model";
import {
  isNullOrUndefined,
  isWhiteString,
} from "../tools";

export class DateUtil {
  private static readonly LOCAL_DATE_DATE_SIMPLE: string = "YYYY-MM-DD";
  private static readonly OFFSET_DATE_TIME_DATE_ISO8601: string = "YYYY-MM-DDTHH:mm:ss.SSSZZ";
  private static readonly FORMAT_TIME_ONLY: string = "LTS";
  private static readonly FORMAT_DATE_ONLY: string = "L";

  static convertToLocalDateDateSimple(date: Date | string): string | null {
    if (isNullOrUndefined(date) || isWhiteString(date)) {
      return null;
    }
    const momentDate = MomentDate(date);
    if (!momentDate.isValid()) {
      return null;
    }
    return momentDate.format(this.LOCAL_DATE_DATE_SIMPLE);
  }

  static convertToOffsetDateTimeIso8601(date: Date | string): string | null {
    // expected format "2019-07-27T00:00:00.000+0200"
    if (isNullOrUndefined(date) || isWhiteString(date)) {
      return null;
    }
    const momentDate = MomentDate(date);
    if (!momentDate.isValid()) {
      return null;
    }
    return momentDate.format(this.OFFSET_DATE_TIME_DATE_ISO8601);
  }

  static convertOffsetDateTimeIso8601ToDate(dateTimeOffsetIso8601: string): Date | null {
    // date format received "2019-07-27T00:00:00.000+0200"
    if (isNullOrUndefined(dateTimeOffsetIso8601) || isWhiteString(dateTimeOffsetIso8601)) {
      return null;
    }
    const dateParsed = Date.parse(dateTimeOffsetIso8601);
    if (isNaN(dateParsed)) {
      return null;
    }
    return new Date(dateParsed);
  }

  static convertDateToDateString(date: Date, defaultValue: string = ""): string {
    if (isNullOrUndefined(date)) {
      return defaultValue;
    }
    return MomentDate(date).format(this.FORMAT_DATE_ONLY);
  }

  static convertDateToTimeString(date: Date, defaultValue: string = ""): string {
    if (isNullOrUndefined(date)) {
      return defaultValue;
    }
    return MomentDate(date).format(this.FORMAT_TIME_ONLY);
  }

  static convertDateToDateTimeString(date: Date, defaultValue: string = ""): string {
    if (isNullOrUndefined(date)) {
      return defaultValue;
    }
    return MomentDate(date).format(this.FORMAT_DATE_ONLY) + " " + MomentDate(date).format(this.FORMAT_TIME_ONLY);
  }

  static extractYearFromDate(date: Date | string): string | null {
    const convertedDate = DateUtil.convertToLocalDateDateSimple(date);
    if (isNullOrUndefined(convertedDate)) {
      return null;
    }
    return MomentDate(convertedDate).year().toString();
  }
}
