import {Type} from "@angular/core";
import {Navigate} from "@ngxs/router-plugin";
import {
  Actions,
  ofActionCompleted,
  StateContext,
  Store,
} from "@ngxs/store";
import {
  Observable,
  pipe,
} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {RegisterDefaultAction} from "../../decorators";
import {SolidifyStateError} from "../../errors/solidify-state.error";
import {ApiService} from "../../http/api.service";
import {
  BaseResourceType,
  BaseStateModel,
  CollectionTyped,
  QueryParameters,
} from "../../models";
import {NotifierService} from "../../models/services/notifier-service.model";
import {
  isEmptyArray,
  isFalse,
  isNullOrUndefined,
  isTrue,
} from "../../tools";
import {MemoizedUtil} from "../../utils";
import {StoreUtil} from "../../utils/stores/store.util";
import {BaseOptions} from "../base/base-options.model";
import {
  BaseState,
  defaultBaseStateInitValue,
} from "../base/base.state";
import {ResourceActionHelper} from "./resource-action.helper";
import {ResourceNameSpace} from "./resource-namespace.model";
import {ResourceOptions} from "./resource-options.model";
import {ResourceStateModel} from "./resource-state.model";
import {ResourceAction} from "./resource.action";

export const defaultResourceStateInitValue: () => ResourceStateModel<BaseResourceType> = () =>
  ({
    ...defaultBaseStateInitValue(),
    total: 0,
    list: undefined,
    current: undefined,
    queryParameters: new QueryParameters(),
    listTemp: undefined,
    listPendingResId: [],
  });

// @dynamic
export abstract class ResourceState<TStateModel extends BaseStateModel, TResource extends BaseResourceType> extends BaseState<TStateModel> {
  protected readonly _nameSpace: ResourceNameSpace;

  protected constructor(protected apiService: ApiService,
                        protected store: Store,
                        protected notifierService: NotifierService,
                        protected actions$: Actions,
                        protected options: ResourceOptions) {
    super(apiService, store, notifierService, actions$, options, ResourceState);
  }

  protected static getDefaultOptions(): ResourceOptions | any {
    // TODO To fix, seems to not work
    let defaultOptions: ResourceOptions | any = {
      keepCurrentStateAfterCreate: false,
      keepCurrentStateAfterUpdate: false,
      keepCurrentStateAfterDelete: false,
    };
    defaultOptions = Object.assign(BaseState.getDefaultOptions(), defaultOptions);
    return defaultOptions;
  }

  protected static checkOptions(stateName: string, options: BaseOptions): void {
    BaseState.checkOptions(stateName, options);
  }

  // static isLoading<TStateModel extends BaseStateModel>(store: Store, ctor: Type<BaseState<TStateModel>>): Observable<boolean> {
  //   return MemoizedUtil.isLoading(store, ctor);
  // }

  static list<TStateModel extends ResourceStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<ResourceState<TStateModel, TResource>>): Observable<TResource[]> {
    return MemoizedUtil.select(store, ctor, state => state.list, true);
  }

  static listSnapshot<TStateModel extends ResourceStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<ResourceState<TStateModel, TResource>>): TResource[] {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.list);
  }

  static total<TStateModel extends ResourceStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<ResourceState<TStateModel, TResource>>): Observable<number> {
    return MemoizedUtil.select(store, ctor, state => state.total, true);
  }

  static totalSnapshot<TStateModel extends ResourceStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<ResourceState<TStateModel, TResource>>): number {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.total);
  }

  static current<TStateModel extends ResourceStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<ResourceState<TStateModel, TResource>>): Observable<TResource> {
    return MemoizedUtil.select(store, ctor, state => state.current, true);
  }

  static currentSnapshot<TStateModel extends ResourceStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<ResourceState<TStateModel, TResource>>): TResource {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.current);
  }

  static queryParameters<TStateModel extends ResourceStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<ResourceState<TStateModel, TResource>>): Observable<QueryParameters> {
    return MemoizedUtil.select(store, ctor, state => state.queryParameters, true);
  }

  static queryParametersSnapshot<TStateModel extends ResourceStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<ResourceState<TStateModel, TResource>>): QueryParameters {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.queryParameters);
  }

  @RegisterDefaultAction((resourceNameSpace: ResourceNameSpace) => resourceNameSpace.LoadResourceSuccess)
  loadResourceSuccess(ctx: StateContext<ResourceStateModel<TResource>>, action: ResourceAction.LoadResourceSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((resourceNameSpace: ResourceNameSpace) => resourceNameSpace.LoadResourceFail)
  loadResourceFail(ctx: StateContext<ResourceStateModel<TResource>>, action: ResourceAction.LoadResourceFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((resourceNameSpace: ResourceNameSpace) => resourceNameSpace.ChangeQueryParameters)
  changeQueryParameters(ctx: StateContext<ResourceStateModel<TResource>>, action: ResourceAction.ChangeQueryParameters): void {
    ctx.patchState({
      queryParameters: action.queryParameters,
    });
    if (isTrue(action.getAllAfterChange)) {
      ctx.dispatch(ResourceActionHelper.getAll(this._nameSpace, undefined, action.keepCurrentContext));
    }
  }

  @RegisterDefaultAction((resourceNameSpace: ResourceNameSpace) => resourceNameSpace.GetAll)
  getAll(ctx: StateContext<ResourceStateModel<TResource>>, action: ResourceAction.GetAll): Observable<CollectionTyped<TResource>> {
    let reset = {};
    if (!action.keepCurrentContext) {
      reset = {
        list: undefined,
        total: 0,
      };
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      queryParameters: StoreUtil.getQueryParametersToApply(action.queryParameters, ctx),
      ...reset,
    });
    return this.apiService.get<TResource>(this._urlResource, ctx.getState().queryParameters)
      .pipe(
        action.cancelIncomplete ? StoreUtil.cancelUncompleted(ctx, this.actions$, [this._nameSpace.GetAll, Navigate]) : pipe(),
        tap((collection: CollectionTyped<TResource>) => {
          ctx.dispatch(ResourceActionHelper.getAllSuccess<TResource>(this._nameSpace, action, collection));
        }),
        catchError(error => {
          ctx.dispatch(ResourceActionHelper.getAllFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((resourceNameSpace: ResourceNameSpace) => resourceNameSpace.GetAllSuccess)
  getAllSuccess(ctx: StateContext<ResourceStateModel<TResource>>, action: ResourceAction.GetAllSuccess<TResource>): void {
    const queryParameters = StoreUtil.updateQueryParameters(ctx, action.list);

    ctx.patchState({
      list: action.list._data,
      total: action.list._page.totalItems,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      queryParameters,
    });
  }

  @RegisterDefaultAction((resourceNameSpace: ResourceNameSpace) => resourceNameSpace.GetAllFail)
  getAllFail(ctx: StateContext<ResourceStateModel<TResource>>, action: ResourceAction.GetAllFail<TResource>): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((resourceNameSpace: ResourceNameSpace) => resourceNameSpace.GetByListId)
  getByListId(ctx: StateContext<ResourceStateModel<TResource>>, action: ResourceAction.GetByListId): void {
    if (isNullOrUndefined(action.listResId) || isEmptyArray(action.listResId)) {
      return;
    }
    let listTemp = undefined;
    let reset = {};
    if (!action.keepCurrentContext) {
      reset = {
        list: undefined,
        listTemp: undefined,
      };
    } else {
      listTemp = ctx.getState().list;
    }

    const listSubAction = [];
    action.listResId.forEach(resId => {
      if (!isNullOrUndefined(listTemp) && !isEmptyArray(listTemp)) {
        const existingItem = listTemp.find(item => item.resId === resId);
        if (!isNullOrUndefined(existingItem)) {
          return;
        }
      }
      listSubAction.push({
        action: ResourceActionHelper.getById(this._nameSpace, resId, true, true),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(this._nameSpace.GetByIdSuccess)),
          this.actions$.pipe(ofActionCompleted(this._nameSpace.GetByIdFail)),
        ],
      });
    });

    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      listTemp: listTemp,
      ...reset,
    });

    StoreUtil.dispatchParallelActionAndWaitForSubActionsCompletion(ctx, listSubAction).subscribe(success => {
      if (success) {
        ctx.dispatch(ResourceActionHelper.getByListIdSuccess(this._nameSpace, action));
      } else {
        ctx.dispatch(ResourceActionHelper.getByListIdFail(this._nameSpace, action));
      }
    });
  }

  @RegisterDefaultAction((resourceNameSpace: ResourceNameSpace) => resourceNameSpace.GetByListIdSuccess)
  getByListIdSuccess(ctx: StateContext<ResourceStateModel<TResource>>, action: ResourceAction.GetByListIdSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      list: ctx.getState().listTemp,
      listTemp: undefined,
    });
  }

  @RegisterDefaultAction((resourceNameSpace: ResourceNameSpace) => resourceNameSpace.GetByListIdFail)
  getByListIdFail(ctx: StateContext<ResourceStateModel<TResource>>, action: ResourceAction.GetByListIdFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      listTemp: undefined,
    });
  }

  @RegisterDefaultAction((resourceNameSpace: ResourceNameSpace) => resourceNameSpace.GetById)
  getById(ctx: StateContext<ResourceStateModel<TResource>>, action: ResourceAction.GetById): Observable<TResource> {
    let reset = {};
    if (!action.keepCurrentContext) {
      reset = {
        current: undefined,
      };
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      ...reset,
    });

    return this.apiService.getById<TResource>(this._urlResource, action.id)
      .pipe(
        tap((model: TResource) => {
          ctx.dispatch(ResourceActionHelper.getByIdSuccess(this._nameSpace, action, model));
        }),
        catchError(error => {
          ctx.dispatch(ResourceActionHelper.getByIdFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((resourceNameSpace: ResourceNameSpace) => resourceNameSpace.GetByIdSuccess)
  getByIdSuccess(ctx: StateContext<ResourceStateModel<TResource>>, action: ResourceAction.GetByIdSuccess<TResource>): void {
    if (!isNullOrUndefined(action.parentAction) && isTrue(action.parentAction.addInListTemp)) {
      let list = ctx.getState().listTemp;
      if (isNullOrUndefined(list)) {
        list = [];
      }
      list = [...list, action.model];
      ctx.patchState({
        listTemp: list,
        isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      });
      return;
    }

    ctx.patchState({
      current: action.model,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((resourceNameSpace: ResourceNameSpace) => resourceNameSpace.GetByIdFail)
  getByIdFail(ctx: StateContext<ResourceStateModel<TResource>>, action: ResourceAction.GetByIdFail<TResource>): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((resourceNameSpace: ResourceNameSpace) => resourceNameSpace.Create)
  create(ctx: StateContext<ResourceStateModel<TResource>>, action: ResourceAction.Create<TResource>): Observable<TResource> {
    return this.internalCreate(ctx, action)
      .pipe(
        tap((model: TResource) => {
          ctx.dispatch(ResourceActionHelper.createSuccess(this._nameSpace, action, model));
        }),
      );
  }

  protected internalCreate(ctx: StateContext<ResourceStateModel<TResource>>, action: ResourceAction.Create<TResource>): Observable<TResource> {
    this.cleanCurrentStateIfDefined(ctx, this.options.keepCurrentStateBeforeCreate);
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return this.apiService.post<TResource>(this._urlResource, action.modelFormControlEvent.model)
      .pipe(
        catchError(error => {
          ctx.dispatch(ResourceActionHelper.createFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
        StoreUtil.catchValidationErrors(ctx, action.modelFormControlEvent),
      );
  }

  @RegisterDefaultAction((resourceNameSpace: ResourceNameSpace) => resourceNameSpace.CreateSuccess)
  createSuccess(ctx: StateContext<ResourceStateModel<TResource>>, action: ResourceAction.CreateSuccess<TResource>): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      current: action.model,
    });

    StoreUtil.navigateIfDefined(ctx, this.options.routeRedirectUrlAfterSuccessCreateAction, action.model.resId);
    StoreUtil.notifySuccess(this.notifierService, this.options.notificationResourceCreateSuccessTextToTranslate);
    this.cleanCurrentStateIfDefined(ctx, this.options.keepCurrentStateAfterCreate);
  }

  @RegisterDefaultAction((resourceNameSpace: ResourceNameSpace) => resourceNameSpace.CreateFail)
  createFail(ctx: StateContext<ResourceStateModel<TResource>>, action: ResourceAction.CreateFail<TResource>): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    StoreUtil.notifyError(this.notifierService, this.options.notificationResourceCreateFailTextToTranslate);
  }

  @RegisterDefaultAction((resourceNameSpace: ResourceNameSpace) => resourceNameSpace.Update)
  update(ctx: StateContext<ResourceStateModel<TResource>>, action: ResourceAction.Update<TResource>): Observable<TResource> {
    return this.internalUpdate(ctx, action)
      .pipe(
        tap((result) => {
          ctx.dispatch(ResourceActionHelper.updateSuccess(this._nameSpace, action, result));
        }),
      );
  }

  protected internalUpdate(ctx: StateContext<ResourceStateModel<TResource>>, action: ResourceAction.Update<TResource>): Observable<TResource> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return this.apiService.patchById<TResource>(this._urlResource, action.modelFormControlEvent.model.resId, action.modelFormControlEvent.model)
      .pipe(
        catchError(error => {
          ctx.dispatch(ResourceActionHelper.updateFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
        StoreUtil.catchValidationErrors(ctx, action.modelFormControlEvent),
      );
  }

  @RegisterDefaultAction((resourceNameSpace: ResourceNameSpace) => resourceNameSpace.UpdateSuccess)
  updateSuccess(ctx: StateContext<ResourceStateModel<TResource>>, action: ResourceAction.UpdateSuccess<TResource>): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      current: action.model,
    });

    StoreUtil.navigateIfDefined(ctx, this.options.routeRedirectUrlAfterSuccessUpdateAction, action.model.resId);
    StoreUtil.notifySuccess(this.notifierService, this.options.notificationResourceUpdateSuccessTextToTranslate);
    this.cleanCurrentStateIfDefined(ctx, this.options.keepCurrentStateAfterUpdate);
  }

  @RegisterDefaultAction((resourceNameSpace: ResourceNameSpace) => resourceNameSpace.UpdateFail)
  updateFail(ctx: StateContext<ResourceStateModel<TResource>>, action: ResourceAction.UpdateFail<TResource>): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    StoreUtil.notifyError(this.notifierService, this.options.notificationResourceUpdateFailTextToTranslate);
  }

  @RegisterDefaultAction((resourceNameSpace: ResourceNameSpace) => resourceNameSpace.Delete)
  delete(ctx: StateContext<ResourceStateModel<TResource>>, action: ResourceAction.Delete): Observable<TResource> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return this.apiService.deleteById<TResource>(this._urlResource, action.resId)
      .pipe(
        tap(() => {
          ctx.dispatch(ResourceActionHelper.deleteSuccess(this._nameSpace, action));
        }),
        catchError(error => {
          ctx.dispatch(ResourceActionHelper.deleteFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((resourceNameSpace: ResourceNameSpace) => resourceNameSpace.DeleteSuccess)
  deleteSuccess(ctx: StateContext<ResourceStateModel<TResource>>, action: ResourceAction.DeleteSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });

    StoreUtil.navigateIfDefined(ctx, this.options.routeRedirectUrlAfterSuccessDeleteAction, undefined);
    StoreUtil.notifySuccess(this.notifierService, this.options.notificationResourceDeleteSuccessTextToTranslate);
    this.cleanCurrentStateIfDefined(ctx, this.options.keepCurrentStateAfterDelete);
  }

  @RegisterDefaultAction((resourceNameSpace: ResourceNameSpace) => resourceNameSpace.DeleteFail)
  deleteFail(ctx: StateContext<ResourceStateModel<TResource>>, action: ResourceAction.DeleteFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    StoreUtil.notifyError(this.notifierService, this.options.notificationResourceDeleteFailTextToTranslate);
  }

  @RegisterDefaultAction((resourceNameSpace: ResourceNameSpace) => resourceNameSpace.AddInList)
  addInList(ctx: StateContext<ResourceStateModel<TResource>>, action: ResourceAction.AddInList<TResource>): void {
    if (isNullOrUndefined(action.model)) {
      return;
    }
    let list = ctx.getState().list;
    if (isNullOrUndefined(list)) {
      list = [];
    }
    list = [...list, action.model];
    ctx.patchState({
      list: list,
      total: list.length,
    });
  }

  @RegisterDefaultAction((resourceNameSpace: ResourceNameSpace) => resourceNameSpace.AddInListById)
  addInListById(ctx: StateContext<ResourceStateModel<TResource>>, action: ResourceAction.AddInListById): Observable<TResource> {
    let indexAlreadyExisting = -1;
    if (!isNullOrUndefined(ctx.getState().list) && isTrue(action.avoidDuplicate)) {
      indexAlreadyExisting = ctx.getState().list.findIndex(item => item.resId === action.resId);
      if (indexAlreadyExisting !== -1 && isFalse(action.replace)) {
        return;
      }
    }
    if (ctx.getState().listPendingResId.find(pending => pending === action.resId)) {
      return;
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      listPendingResId: [...(isNullOrUndefined(ctx.getState().listPendingResId) ? [] : ctx.getState().listPendingResId), action.resId],
    });
    return this.apiService.getById<TResource>(this._urlResource, action.resId)
      .pipe(
        tap((model: TResource) => {
          ctx.dispatch(ResourceActionHelper.addInListByIdSuccess(this._nameSpace, action, model, indexAlreadyExisting));
        }),
        catchError(error => {
          ctx.dispatch(ResourceActionHelper.addInListByIdFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((resourceNameSpace: ResourceNameSpace) => resourceNameSpace.AddInListByIdSuccess)
  addInListByIdSuccess(ctx: StateContext<ResourceStateModel<TResource>>, action: ResourceAction.AddInListByIdSuccess<TResource>): void {
    if (isNullOrUndefined(action.model)) {
      return;
    }
    let list = ctx.getState().list;
    if (isNullOrUndefined(list)) {
      list = [];
    }
    if (action.indexAlreadyExisting !== -1 && isTrue(action.parentAction.replace)) {
      list = [...list];
      list[action.indexAlreadyExisting] = action.model;
    } else {
      list = [...list, action.model];
    }
    ctx.patchState({
      list: list,
      total: list.length,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      listPendingResId: this.getListPendingResIdWithValueRemoved(ctx, action.parentAction.resId),
    });
  }

  @RegisterDefaultAction((resourceNameSpace: ResourceNameSpace) => resourceNameSpace.AddInListByIdFail)
  addInListByIdFail(ctx: StateContext<ResourceStateModel<TResource>>, action: ResourceAction.AddInListByIdFail<TResource>): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      listPendingResId: this.getListPendingResIdWithValueRemoved(ctx, (action.parentAction as ResourceAction.AddInListById).resId),
    });
  }

  private getListPendingResIdWithValueRemoved(ctx: StateContext<ResourceStateModel<TResource>>, resId: string): string[] {
    let listPendingResId = ctx.getState().listPendingResId;
    const indexOf = listPendingResId.indexOf(resId);
    if (indexOf === -1) {
      return listPendingResId;
    }
    listPendingResId = [...listPendingResId];
    listPendingResId.splice(indexOf, 1);
    return listPendingResId;
  }

  @RegisterDefaultAction((resourceNameSpace: ResourceNameSpace) => resourceNameSpace.RemoveInListById)
  removeInListById(ctx: StateContext<ResourceStateModel<TResource>>, action: ResourceAction.RemoveInListById): void {
    const list = [...ctx.getState().list];
    if (isNullOrUndefined(action.resId) || isNullOrUndefined(list) || isEmptyArray(list)) {
      return;
    }
    const indexOfItemToRemove = list.findIndex(i => i.resId === action.resId);
    if (indexOfItemToRemove === -1) {
      return;
    }
    list.splice(indexOfItemToRemove, 1);
    ctx.patchState({
      list: list,
      total: list.length,
    });
  }

  @RegisterDefaultAction((resourceNameSpace: ResourceNameSpace) => resourceNameSpace.RemoveInListByListId)
  removeInListByListId(ctx: StateContext<ResourceStateModel<TResource>>, action: ResourceAction.RemoveInListByListId): void {
    const list = [...ctx.getState().list];
    if (isNullOrUndefined(action.listResId) || isEmptyArray(action.listResId) || isNullOrUndefined(list) || isEmptyArray(list)) {
      return;
    }
    action.listResId.forEach(resId => {
      const indexOfItemToRemove = list.findIndex(i => i.resId === resId);
      if (indexOfItemToRemove === -1) {
        return; // continue to iterate
      }
      list.splice(indexOfItemToRemove, 1);
    });
    ctx.patchState({
      list: list,
      total: list.length,
    });
  }

  @RegisterDefaultAction((resourceNameSpace: ResourceNameSpace) => resourceNameSpace.Clean)
  clean(ctx: StateContext<ResourceStateModel<TResource>>, action: ResourceAction.Clean): void {
    this.resetStateToDefault(ctx, isNullOrUndefined(action.preserveList) ? true : action.preserveList, action.queryParameter);
  }

  private cleanCurrentStateIfDefined(ctx: StateContext<ResourceStateModel<TResource>>, keepCurrentState: boolean): void {
    if (isNullOrUndefined(keepCurrentState) || isFalse(keepCurrentState)) {
      this.resetStateToDefault(ctx, true);
    }
  }

  private resetStateToDefault(ctx: StateContext<ResourceStateModel<TResource>>, preserveListData: boolean, newQueryParameters?: QueryParameters): ResourceStateModel<BaseResourceType> {
    const list = ctx.getState().list;
    const total = ctx.getState().total;
    const queryParameters = ctx.getState().queryParameters;
    let result = ctx.setState(this.getDefaultData());
    if (isTrue(preserveListData)) {
      result = ctx.patchState({
        list,
        total,
        queryParameters,
      });
    }
    if (!isNullOrUndefined(newQueryParameters)) {
      ctx.patchState({
        queryParameters: newQueryParameters,
      });
    }
    return result;
  }

  private getDefaultData(): ResourceStateModel<BaseResourceType> {
    return StoreUtil.getDefaultData(this.constructor);
  }
}
