import {Type} from "@angular/core";
import {
  Actions,
  StateContext,
  Store,
} from "@ngxs/store";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {RegisterDefaultAction} from "../../decorators/store.decorator";
import {SolidifyStateError} from "../../errors";
import {ApiService} from "../../http/api.service";
import {
  BaseResourceType,
  BaseStateModel,
} from "../../models";
import {CollectionTyped} from "../../models/dto/collection-typed.model";
import {QueryParameters} from "../../models/query-parameters/query-parameters.model";
import {NotifierService} from "../../models/services/notifier-service.model";
import {MemoizedUtil} from "../../utils";
import {StoreUtil} from "../../utils/stores/store.util";
import {
  BaseState,
  defaultBaseStateInitValue,
} from "../base/base.state";
import {ResourceReadOnlyActionHelper} from "./resource-read-only-action.helper";
import {ResourceReadOnlyNameSpace} from "./resource-read-only-namespace.model";
import {ResourceReadOnlyOptions} from "./resource-read-only-options.model";
import {ResourceReadOnlyStateModel} from "./resource-read-only-state.model";
import {ResourceReadOnlyAction} from "./resource-read-only.action";

export const defaultResourceReadOnlyStateInitValue: () => ResourceReadOnlyStateModel<BaseResourceType> = () =>
  ({
    ...defaultBaseStateInitValue(),
    total: 0,
    list: undefined,
    current: undefined,
    queryParameters: new QueryParameters(),
  });

// @dynamic
export abstract class ResourceReadOnlyState<TStateModel extends BaseStateModel, TResource extends BaseResourceType> extends BaseState<TStateModel> {
  protected readonly _nameSpace: ResourceReadOnlyNameSpace;

  protected constructor(protected apiService: ApiService,
                        protected store: Store,
                        protected notifierService: NotifierService,
                        protected actions$: Actions,
                        protected options: ResourceReadOnlyOptions) {
    super(apiService, store, notifierService, actions$, options, ResourceReadOnlyState);
  }

  static list<TStateModel extends ResourceReadOnlyStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<ResourceReadOnlyState<TStateModel, TResource>>): Observable<TResource[]> {
    return MemoizedUtil.select(store, ctor, state => state.list, true);
  }

  static listSnapshot<TStateModel extends ResourceReadOnlyStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<ResourceReadOnlyState<TStateModel, TResource>>): TResource[] {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.list);
  }

  static total<TStateModel extends ResourceReadOnlyStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<ResourceReadOnlyState<TStateModel, TResource>>): Observable<number> {
    return MemoizedUtil.select(store, ctor, state => state.total, true);
  }

  static totalSnapshot<TStateModel extends ResourceReadOnlyStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<ResourceReadOnlyState<TStateModel, TResource>>): number {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.total);
  }

  static current<TStateModel extends ResourceReadOnlyStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<ResourceReadOnlyState<TStateModel, TResource>>): Observable<TResource> {
    return MemoizedUtil.select(store, ctor, state => state.current, true);
  }

  static currentSnapshot<TStateModel extends ResourceReadOnlyStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<ResourceReadOnlyState<TStateModel, TResource>>): TResource {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.current);
  }

  static queryParameters<TStateModel extends ResourceReadOnlyStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<ResourceReadOnlyState<TStateModel, TResource>>): Observable<QueryParameters> {
    return MemoizedUtil.select(store, ctor, state => state.queryParameters, true);
  }

  static queryParametersSnapshot<TStateModel extends ResourceReadOnlyStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<ResourceReadOnlyState<TStateModel, TResource>>): QueryParameters {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.queryParameters);
  }

  @RegisterDefaultAction((resourceReadOnlyNameSpace: ResourceReadOnlyNameSpace) => resourceReadOnlyNameSpace.ChangeQueryParameters)
  changeQueryParameters(ctx: StateContext<ResourceReadOnlyStateModel<TResource>>, action: ResourceReadOnlyAction.ChangeQueryParameters): void {
    ctx.patchState({
      queryParameters: action.queryParameters,
    });
    ctx.dispatch(ResourceReadOnlyActionHelper.getAll(this._nameSpace, undefined, action.keepCurrentContext));
  }

  @RegisterDefaultAction((resourceReadOnlyNameSpace: ResourceReadOnlyNameSpace) => resourceReadOnlyNameSpace.GetAll)
  getAll(ctx: StateContext<ResourceReadOnlyStateModel<TResource>>, action: ResourceReadOnlyAction.GetAll): Observable<CollectionTyped<TResource>> {
    let reset = {};
    if (!action.keepCurrentContext) {
      reset = {
        total: 0,
        list: undefined,
      };
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      queryParameters: StoreUtil.getQueryParametersToApply(action.queryParameters, ctx),
      ...reset,
    });
    return this.apiService.get<TResource>(this._urlResource, ctx.getState().queryParameters)
      .pipe(
        tap((collection: CollectionTyped<TResource>) => {
          ctx.dispatch(ResourceReadOnlyActionHelper.getAllSuccess<TResource>(this._nameSpace, action, collection));
        }),
        catchError(error => {
          ctx.dispatch(ResourceReadOnlyActionHelper.getAllFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((resourceReadOnlyNameSpace: ResourceReadOnlyNameSpace) => resourceReadOnlyNameSpace.GetAllSuccess)
  getAllSuccess(ctx: StateContext<ResourceReadOnlyStateModel<TResource>>, action: ResourceReadOnlyAction.GetAllSuccess<TResource>): void {
    ctx.patchState({
      list: action.list._data,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((resourceReadOnlyNameSpace: ResourceReadOnlyNameSpace) => resourceReadOnlyNameSpace.GetAllFail)
  getAllFail(ctx: StateContext<ResourceReadOnlyStateModel<TResource>>, action: ResourceReadOnlyAction.GetAllFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((resourceReadOnlyNameSpace: ResourceReadOnlyNameSpace) => resourceReadOnlyNameSpace.GetById)
  getById(ctx: StateContext<ResourceReadOnlyStateModel<TResource>>, action: ResourceReadOnlyAction.GetById): Observable<TResource> {
    let reset = {};
    if (!action.keepCurrentContext) {
      reset = {
        current: undefined,
      };
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      ...reset,
    });

    return this.apiService.getById<TResource>(this._urlResource, action.id)
      .pipe(
        tap((model: TResource) => {
          ctx.dispatch(ResourceReadOnlyActionHelper.getByIdSuccess(this._nameSpace, action, model));
        }),
        catchError(error => {
          ctx.dispatch(ResourceReadOnlyActionHelper.getByIdFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((resourceReadOnlyNameSpace: ResourceReadOnlyNameSpace) => resourceReadOnlyNameSpace.GetByIdSuccess)
  getByIdSuccess(ctx: StateContext<ResourceReadOnlyStateModel<TResource>>, action: ResourceReadOnlyAction.GetByIdSuccess<TResource>): void {
    ctx.patchState({
      current: action.model,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((resourceReadOnlyNameSpace: ResourceReadOnlyNameSpace) => resourceReadOnlyNameSpace.GetByIdFail)
  getByIdFail(ctx: StateContext<ResourceReadOnlyStateModel<TResource>>, action: ResourceReadOnlyAction.GetByIdFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }
}
