import {Type} from "@angular/core";
import {
  Actions,
  StateContext,
  Store,
} from "@ngxs/store";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {RegisterDefaultAction} from "../../decorators/store.decorator";
import {SolidifyStateError} from "../../errors";
import {ApiService} from "../../http/api.service";
import {
  BaseResourceType,
  BaseStateModel,
} from "../../models";
import {CollectionTyped} from "../../models/dto/collection-typed.model";
import {QueryParameters} from "../../models/query-parameters/query-parameters.model";
import {NotifierService} from "../../models/services/notifier-service.model";
import {MemoizedUtil} from "../../utils";
import {StoreUtil} from "../../utils/stores/store.util";
import {
  BaseState,
  defaultBaseStateInitValue,
} from "../base/base.state";
import {ResourceNoSqlReadOnlyActionHelper} from "./resource-no-sql-read-only-action.helper";
import {ResourceNoSqlReadOnlyNameSpace} from "./resource-no-sql-read-only-namespace.model";
import {ResourceNoSqlReadOnlyOptions} from "./resource-no-sql-read-only-options.model";
import {ResourceNoSqlReadOnlyStateModel} from "./resource-no-sql-read-only-state.model";
import {ResourceNoSqlReadOnlyAction} from "./resource-no-sql-read-only.action";

export const defaultResourceNoSqlReadOnlyStateInitValue: () => ResourceNoSqlReadOnlyStateModel<BaseResourceType> = () =>
  ({
    ...defaultBaseStateInitValue(),
    total: 0,
    list: undefined,
    current: undefined,
    queryParameters: new QueryParameters(),
  });

// @dynamic
export abstract class ResourceNoSqlReadOnlyState<TStateModel extends BaseStateModel, TResource extends BaseResourceType> extends BaseState<TStateModel> {
  protected readonly _nameSpace: ResourceNoSqlReadOnlyNameSpace;

  protected constructor(protected apiService: ApiService,
                        protected store: Store,
                        protected notifierService: NotifierService,
                        protected actions$: Actions,
                        protected options: ResourceNoSqlReadOnlyOptions) {
    super(apiService, store, notifierService, actions$, options, ResourceNoSqlReadOnlyState);
  }

  static list<TStateModel extends ResourceNoSqlReadOnlyStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<ResourceNoSqlReadOnlyState<TStateModel, TResource>>): Observable<TResource[]> {
    return MemoizedUtil.select(store, ctor, state => state.list, true);
  }

  static listSnapshot<TStateModel extends ResourceNoSqlReadOnlyStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<ResourceNoSqlReadOnlyState<TStateModel, TResource>>): TResource[] {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.list);
  }

  static total<TStateModel extends ResourceNoSqlReadOnlyStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<ResourceNoSqlReadOnlyState<TStateModel, TResource>>): Observable<number> {
    return MemoizedUtil.select(store, ctor, state => state.total, true);
  }

  static totalSnapshot<TStateModel extends ResourceNoSqlReadOnlyStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<ResourceNoSqlReadOnlyState<TStateModel, TResource>>): number {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.total);
  }

  static current<TStateModel extends ResourceNoSqlReadOnlyStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<ResourceNoSqlReadOnlyState<TStateModel, TResource>>): Observable<TResource> {
    return MemoizedUtil.select(store, ctor, state => state.current, true);
  }

  static currentSnapshot<TStateModel extends ResourceNoSqlReadOnlyStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<ResourceNoSqlReadOnlyState<TStateModel, TResource>>): TResource {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.current);
  }

  static queryParameters<TStateModel extends ResourceNoSqlReadOnlyStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<ResourceNoSqlReadOnlyState<TStateModel, TResource>>): Observable<QueryParameters> {
    return MemoizedUtil.select(store, ctor, state => state.queryParameters, true);
  }

  static queryParametersSnapshot<TStateModel extends ResourceNoSqlReadOnlyStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<ResourceNoSqlReadOnlyState<TStateModel, TResource>>): QueryParameters {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.queryParameters);
  }

  @RegisterDefaultAction((resourceNoSqlReadOnlyNameSpace: ResourceNoSqlReadOnlyNameSpace) => resourceNoSqlReadOnlyNameSpace.ChangeQueryParameters)
  changeQueryParameters(ctx: StateContext<ResourceNoSqlReadOnlyStateModel<TResource>>, action: ResourceNoSqlReadOnlyAction.ChangeQueryParameters): void {
    ctx.patchState({
      queryParameters: action.queryParameters,
    });
    ctx.dispatch(ResourceNoSqlReadOnlyActionHelper.getAll(this._nameSpace, undefined, action.keepCurrentContext));
  }

  @RegisterDefaultAction((resourceNoSqlReadOnlyNameSpace: ResourceNoSqlReadOnlyNameSpace) => resourceNoSqlReadOnlyNameSpace.GetAll)
  getAll(ctx: StateContext<ResourceNoSqlReadOnlyStateModel<TResource>>, action: ResourceNoSqlReadOnlyAction.GetAll): Observable<CollectionTyped<TResource>> {
    let reset = {};
    if (!action.keepCurrentContext) {
      reset = {
        total: 0,
        list: undefined,
      };
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      queryParameters: StoreUtil.getQueryParametersToApply(action.queryParameters, ctx),
      ...reset,
    });
    return this.apiService.get<TResource>(this._urlResource, ctx.getState().queryParameters)
      .pipe(
        tap((collection: CollectionTyped<TResource>) => {
          ctx.dispatch(ResourceNoSqlReadOnlyActionHelper.getAllSuccess<TResource>(this._nameSpace, action, collection));
        }),
        catchError(error => {
          ctx.dispatch(ResourceNoSqlReadOnlyActionHelper.getAllFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((resourceNoSqlReadOnlyNameSpace: ResourceNoSqlReadOnlyNameSpace) => resourceNoSqlReadOnlyNameSpace.GetAllSuccess)
  getAllSuccess(ctx: StateContext<ResourceNoSqlReadOnlyStateModel<TResource>>, action: ResourceNoSqlReadOnlyAction.GetAllSuccess<TResource>): void {
    ctx.patchState({
      list: action.list._data,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((resourceNoSqlReadOnlyNameSpace: ResourceNoSqlReadOnlyNameSpace) => resourceNoSqlReadOnlyNameSpace.GetAllFail)
  getAllFail(ctx: StateContext<ResourceNoSqlReadOnlyStateModel<TResource>>, action: ResourceNoSqlReadOnlyAction.GetAllFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((resourceNoSqlReadOnlyNameSpace: ResourceNoSqlReadOnlyNameSpace) => resourceNoSqlReadOnlyNameSpace.GetById)
  getById(ctx: StateContext<ResourceNoSqlReadOnlyStateModel<TResource>>, action: ResourceNoSqlReadOnlyAction.GetById): Observable<TResource> {
    let reset = {};
    if (!action.keepCurrentContext) {
      reset = {
        current: undefined,
      };
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      ...reset,
    });

    return this.apiService.getById<TResource>(this._urlResource, action.id)
      .pipe(
        tap((model: TResource) => {
          ctx.dispatch(ResourceNoSqlReadOnlyActionHelper.getByIdSuccess(this._nameSpace, action, model));
        }),
        catchError(error => {
          ctx.dispatch(ResourceNoSqlReadOnlyActionHelper.getByIdFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((resourceNoSqlReadOnlyNameSpace: ResourceNoSqlReadOnlyNameSpace) => resourceNoSqlReadOnlyNameSpace.GetByIdSuccess)
  getByIdSuccess(ctx: StateContext<ResourceNoSqlReadOnlyStateModel<TResource>>, action: ResourceNoSqlReadOnlyAction.GetByIdSuccess<TResource>): void {
    ctx.patchState({
      current: action.model,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((resourceNoSqlReadOnlyNameSpace: ResourceNoSqlReadOnlyNameSpace) => resourceNoSqlReadOnlyNameSpace.GetByIdFail)
  getByIdFail(ctx: StateContext<ResourceNoSqlReadOnlyStateModel<TResource>>, action: ResourceNoSqlReadOnlyAction.GetByIdFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }
}
