import {Type} from "@angular/core";
import {
  Actions,
  ofActionCompleted,
  StateContext,
  Store,
} from "@ngxs/store";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {RegisterDefaultAction} from "../../decorators/store.decorator";
import {ApiResourceNameEnum} from "../../enums/partial/api-resource-name.enum";
import {urlSeparator} from "../../enums/partial/routes.enum";
import {SolidifyStateError} from "../../errors";
import {ApiService} from "../../http/api.service";
import {
  ActionSubActionCompletionsWrapper,
  BaseResourceType,
  BaseStateModel,
} from "../../models";
import {BaseRelationResourceType} from "../../models/dto/base-relation-resource.model";
import {CollectionTyped} from "../../models/dto/collection-typed.model";
import {QueryParameters} from "../../models/query-parameters/query-parameters.model";
import {NotifierService} from "../../models/services/notifier-service.model";
import {SubResourceUpdateModel} from "../../models/stores/sub-resource-update.model";
import {
  isEmptyArray,
  isNullOrUndefined,
  isUndefined,
} from "../../tools";
import {MemoizedUtil} from "../../utils";
import {StoreUtil} from "../../utils/stores/store.util";
import {
  BaseState,
  defaultBaseStateInitValue,
} from "../base/base.state";
import {Relation3TiersActionHelper} from "./relation-3-tiers-action.helper";
import {Relation3TiersForm} from "./relation-3-tiers-form.model";
import {Relation3TiersNameSpace} from "./relation-3-tiers-namespace.model";
import {Relation3TiersOptions} from "./relation-3-tiers-options.model";
import {Relation3TiersStateModel} from "./relation-3-tiers-state.model";
import {Relation3TiersAction} from "./relation-3-tiers.action";

export const defaultRelation3TiersStateInitValue: () => Relation3TiersStateModel<BaseResourceType> = () =>
  ({
    ...defaultBaseStateInitValue(),
    current: undefined,
    total: 0,
    selected: undefined,
    queryParameters: new QueryParameters(),
  });

// @dynamic
export abstract class Relation3TiersState<TStateModel extends BaseStateModel, TResource extends BaseResourceType, TRelation extends BaseRelationResourceType> extends BaseState<TStateModel> {
  protected readonly _resourceName: ApiResourceNameEnum;
  protected readonly _nameSpace: Relation3TiersNameSpace;

  protected constructor(protected apiService: ApiService,
                        protected store: Store,
                        protected notifierService: NotifierService,
                        protected actions$: Actions,
                        protected options: Relation3TiersOptions) {
    super(apiService, store, notifierService, actions$, options, Relation3TiersState);
    this._resourceName = this.options.resourceName;
  }

  static total<TStateModel extends Relation3TiersStateModel<TResource>, TResource extends BaseResourceType, TRelation extends BaseRelationResourceType>(store: Store, ctor: Type<Relation3TiersState<TStateModel, TResource, TRelation>>): Observable<number> {
    return MemoizedUtil.select(store, ctor, state => state.total, true);
  }

  static totalSnapshot<TStateModel extends Relation3TiersStateModel<TResource>, TResource extends BaseResourceType, TRelation extends BaseRelationResourceType>(store: Store, ctor: Type<Relation3TiersState<TStateModel, TResource, TRelation>>): number {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.total);
  }

  static current<TStateModel extends Relation3TiersStateModel<TResource>, TResource extends BaseResourceType, TRelation extends BaseRelationResourceType>(store: Store, ctor: Type<Relation3TiersState<TStateModel, TResource, TRelation>>): Observable<TResource> {
    return MemoizedUtil.select(store, ctor, state => state.current, true);
  }

  static currentSnapshot<TStateModel extends Relation3TiersStateModel<TResource>, TResource extends BaseResourceType, TRelation extends BaseRelationResourceType>(store: Store, ctor: Type<Relation3TiersState<TStateModel, TResource, TRelation>>): TResource {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.current);
  }

  static selected<TStateModel extends Relation3TiersStateModel<TResource>, TResource extends BaseResourceType, TRelation extends BaseRelationResourceType>(store: Store, ctor: Type<Relation3TiersState<TStateModel, TResource, TRelation>>): Observable<TResource[]> {
    return MemoizedUtil.select(store, ctor, state => state.selected, true);
  }

  static selectedSnapshot<TStateModel extends Relation3TiersStateModel<TResource>, TResource extends BaseResourceType, TRelation extends BaseRelationResourceType>(store: Store, ctor: Type<Relation3TiersState<TStateModel, TResource, TRelation>>): TResource[] {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.selected);
  }

  static queryParameters<TStateModel extends Relation3TiersStateModel<TResource>, TResource extends BaseResourceType, TRelation extends BaseRelationResourceType>(store: Store, ctor: Type<Relation3TiersState<TStateModel, TResource, TRelation>>): Observable<QueryParameters> {
    return MemoizedUtil.select(store, ctor, state => state.queryParameters, true);
  }

  static queryParametersSnapshot<TStateModel extends Relation3TiersStateModel<TResource>, TResource extends BaseResourceType, TRelation extends BaseRelationResourceType>(store: Store, ctor: Type<Relation3TiersState<TStateModel, TResource, TRelation>>): QueryParameters {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.queryParameters);
  }

  private evaluateSubResourceUrl(parentId: string, id: string | undefined = undefined): string {
    const url = this._urlResource + urlSeparator + parentId + urlSeparator + this._resourceName;
    if (isUndefined(id)) {
      return url;
    }
    return url + urlSeparator + id;
  }

  protected abstract convertResourceInForm(resource: TResource): Relation3TiersForm;

  private convertSelectedToForm(list: TResource[]): Relation3TiersForm[] {
    const form = [];
    if (isNullOrUndefined(list)) {
      return [];
    }
    list.forEach(r => {
      form.push(this.convertResourceInForm(r));
    });
    return form;
  }

  @RegisterDefaultAction((relation3TiersNameSpace: Relation3TiersNameSpace) => relation3TiersNameSpace.GetAll)
  getAll<U>(ctx: StateContext<Relation3TiersStateModel<TResource>>, action: Relation3TiersAction.GetAll): Observable<CollectionTyped<U>> {
    let reset = {};
    if (!action.keepCurrentContext) {
      reset = {
        total: 0,
        selected: undefined,
      };
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      queryParameters: StoreUtil.getQueryParametersToApply(action.queryParameters, ctx),
      ...reset,
    });
    const url = this.evaluateSubResourceUrl(action.parentId);
    return this.apiService.get<U>(url, ctx.getState().queryParameters)
      .pipe(
        tap((collection: CollectionTyped<U>) => {
          ctx.dispatch(Relation3TiersActionHelper.getAllSuccess<U>(this._nameSpace, action, collection));
        }),
        catchError(error => {
          ctx.dispatch(Relation3TiersActionHelper.getAllFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((relation3TiersNameSpace: Relation3TiersNameSpace) => relation3TiersNameSpace.GetAllSuccess)
  getAllSuccess(ctx: StateContext<Relation3TiersStateModel<TResource>>, action: Relation3TiersAction.GetAllSuccess<TResource>): void {
    ctx.patchState({
      selected: action.list._data,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((relation3TiersNameSpace: Relation3TiersNameSpace) => relation3TiersNameSpace.GetAllFail)
  getAllFail(ctx: StateContext<Relation3TiersStateModel<TResource>>, action: Relation3TiersAction.GetAllFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((relation3TiersNameSpace: Relation3TiersNameSpace) => relation3TiersNameSpace.GetById)
  getById<U>(ctx: StateContext<Relation3TiersStateModel<TResource>>, action: Relation3TiersAction.GetById): Observable<U> {
    let reset = {};
    if (!action.keepCurrentContext) {
      reset = {
        current: undefined,
      };
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      queryParameters: StoreUtil.getQueryParametersToApply(action.queryParameters, ctx),
      ...reset,
    });
    const url = this.evaluateSubResourceUrl(action.parentId, action.id);
    return this.apiService.get<U>(url, undefined)
      .pipe(
        tap((model: U) => {
          ctx.dispatch(Relation3TiersActionHelper.getByIdSuccess<U>(this._nameSpace, action, model));
        }),
        catchError(error => {
          ctx.dispatch(Relation3TiersActionHelper.getByIdFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((relation3TiersNameSpace: Relation3TiersNameSpace) => relation3TiersNameSpace.GetByIdSuccess)
  getByIdSuccess(ctx: StateContext<Relation3TiersStateModel<TResource>>, action: Relation3TiersAction.GetByIdSuccess<TResource>): void {
    ctx.patchState({
      current: action.current,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((relation3TiersNameSpace: Relation3TiersNameSpace) => relation3TiersNameSpace.GetByIdFail)
  getByIdFail(ctx: StateContext<Relation3TiersStateModel<TResource>>, action: Relation3TiersAction.GetByIdFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((relation3TiersNameSpace: Relation3TiersNameSpace) => relation3TiersNameSpace.Create)
  create(ctx: StateContext<Relation3TiersStateModel<TResource>>, action: Relation3TiersAction.Create): Observable<string[]> {
    if (action.listResId.length === 0) {
      return;
    }
    const url = this.evaluateSubResourceUrl(action.parentId, action.id);
    return this.apiService.post<string[]>(url, action.listResId)
      .pipe(
        tap(() => {
          ctx.dispatch(Relation3TiersActionHelper.createSuccess(this._nameSpace, action));
        }),
        catchError(error => {
          ctx.dispatch(Relation3TiersActionHelper.createFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((relation3TiersNameSpace: Relation3TiersNameSpace) => relation3TiersNameSpace.CreateResource)
  createResource(ctx: StateContext<Relation3TiersStateModel<TResource>>, action: Relation3TiersAction.CreateResource): Observable<string[]> {
    if (action.listResId.length === 0) {
      return;
    }
    const url = this.evaluateSubResourceUrl(action.parentId);
    return this.apiService.post<string[]>(url, action.listResId)
      .pipe(
        tap(() => {
          ctx.dispatch(Relation3TiersActionHelper.createResourceSuccess(this._nameSpace, action));
        }),
        catchError(error => {
          ctx.dispatch(Relation3TiersActionHelper.createResourceFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((relation3TiersNameSpace: Relation3TiersNameSpace) => relation3TiersNameSpace.Delete)
  delete(ctx: StateContext<Relation3TiersStateModel<TResource>>, action: Relation3TiersAction.Delete): Observable<string[]> {
    const url = this.evaluateSubResourceUrl(action.parentId, action.id);
    return this.apiService.delete<string[]>(url, action.listResId)
      .pipe(
        tap(() => {
          ctx.dispatch(Relation3TiersActionHelper.deleteSuccess(this._nameSpace, action));
        }),
        catchError(error => {
          ctx.dispatch(Relation3TiersActionHelper.deleteFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((relation3TiersNameSpace: Relation3TiersNameSpace) => relation3TiersNameSpace.Update)
  update(ctx: StateContext<Relation3TiersStateModel<TResource>>, action: Relation3TiersAction.Update): Observable<boolean> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    const listOldForm = this.convertSelectedToForm(ctx.getState().selected);
    const listNewForm = action.newForm;

    const subResourceUpdate: SubResourceUpdateModel = StoreUtil.determineSubResourceChange(listOldForm.map(o => o.id), listNewForm.map(o => o.id));
    const createAndUpdateActions = this.computeResourceUpdatesActionsToDispatch(subResourceUpdate, action.parentId);

    listNewForm.forEach(newForm => {
      const oldForm: Relation3TiersForm = listOldForm.find(old => old.id === newForm.id);
      let relationActionSubActionCompletionsWrapper: ActionSubActionCompletionsWrapper[];
      if (isNullOrUndefined(oldForm)) {
        relationActionSubActionCompletionsWrapper = this.computeRelationUpdatesActionsToDispatch({resourceToAdd: newForm.listId} as SubResourceUpdateModel, action.parentId, newForm.id);
      } else {
        const subResourceListUpdate: SubResourceUpdateModel = StoreUtil.determineSubResourceChange(oldForm.listId, newForm.listId);
        relationActionSubActionCompletionsWrapper = this.computeRelationUpdatesActionsToDispatch(subResourceListUpdate, action.parentId, newForm.id);
      }
      createAndUpdateActions.push(...relationActionSubActionCompletionsWrapper);
    });

    // console.debug("Dispatch sequential", createAndUpdateActions);
    return StoreUtil.dispatchSequentialActionAndWaitForSubActionsCompletion(ctx, createAndUpdateActions)
      .pipe(
        tap(success => {
          // console.debug("Sequential finish with status", success);
          if (success) {
            ctx.dispatch(Relation3TiersActionHelper.updateSuccess(this._nameSpace, action, action.parentId));
          } else {
            ctx.dispatch(Relation3TiersActionHelper.updateFail(this._nameSpace, action, action.parentId));
          }
        }),
      );
  }

  private computeResourceUpdatesActionsToDispatch(subResourceUpdate: SubResourceUpdateModel, parentId: string): ActionSubActionCompletionsWrapper[] {
    const actions = [];
    if (!isNullOrUndefined(subResourceUpdate.resourceToRemoved)) {
      subResourceUpdate.resourceToRemoved.forEach(id => {
        const deleteAction = Relation3TiersActionHelper.delete(this._nameSpace, parentId, id, null);
        actions.push({
          action: deleteAction,
          subActionCompletions: [
            this.actions$.pipe(ofActionCompleted(this._nameSpace.DeleteSuccess)),
            this.actions$.pipe(ofActionCompleted(this._nameSpace.DeleteFail)),
          ],
        });
      });
    }
    if (!isNullOrUndefined(subResourceUpdate.resourceToAdd) && !isEmptyArray(subResourceUpdate.resourceToAdd)) {
      const createAction = Relation3TiersActionHelper.createResource(this._nameSpace, parentId, subResourceUpdate.resourceToAdd);
      actions.push({
        action: createAction,
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(this._nameSpace.CreateResourceSuccess)),
          this.actions$.pipe(ofActionCompleted(this._nameSpace.CreateResourceFail)),
        ],
      });
    }
    return actions;
  }

  private computeRelationUpdatesActionsToDispatch(subResourceUpdate: SubResourceUpdateModel, parentId: string, id: string): ActionSubActionCompletionsWrapper[] {
    const actions = [];
    if (!isNullOrUndefined(subResourceUpdate.resourceToRemoved) && !isEmptyArray(subResourceUpdate.resourceToRemoved)) {
      const deleteAction = Relation3TiersActionHelper.delete(this._nameSpace, parentId, id, subResourceUpdate.resourceToRemoved);
      actions.push({
        action: deleteAction,
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(this._nameSpace.DeleteSuccess)),
          this.actions$.pipe(ofActionCompleted(this._nameSpace.DeleteFail)),
        ],
      });
    }
    if (!isNullOrUndefined(subResourceUpdate.resourceToAdd) && !isEmptyArray(subResourceUpdate.resourceToAdd)) {
      const createAction = Relation3TiersActionHelper.create(this._nameSpace, parentId, id, subResourceUpdate.resourceToAdd);
      actions.push({
        action: createAction,
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(this._nameSpace.CreateSuccess)),
          this.actions$.pipe(ofActionCompleted(this._nameSpace.CreateFail)),
        ],
      });
    }
    return actions;
  }

  @RegisterDefaultAction((relation3TiersNameSpace: Relation3TiersNameSpace) => relation3TiersNameSpace.UpdateSuccess)
  updateSuccess(ctx: StateContext<Relation3TiersStateModel<TResource>>, action: Relation3TiersAction.UpdateSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      selected: undefined,
    });
  }

  @RegisterDefaultAction((relation3TiersNameSpace: Relation3TiersNameSpace) => relation3TiersNameSpace.UpdateFail)
  updateFail(ctx: StateContext<Relation3TiersStateModel<TResource>>, action: Relation3TiersAction.UpdateFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    ctx.dispatch(Relation3TiersActionHelper.getAll(this._nameSpace, action.parentId));
  }
}
