import {Type} from "@angular/core";
import {
  Actions,
  ofActionCompleted,
  StateContext,
  Store,
} from "@ngxs/store";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {RegisterDefaultAction} from "../../decorators/store.decorator";
import {ApiResourceNameEnum} from "../../enums/partial/api-resource-name.enum";
import {urlSeparator} from "../../enums/partial/routes.enum";
import {SolidifyStateError} from "../../errors";
import {ApiService} from "../../http/api.service";
import {BaseStateModel} from "../../models";
import {BaseRelationResourceType} from "../../models/dto/base-relation-resource.model";
import {BaseResourceType} from "../../models/dto/base-resource.model";
import {CollectionTyped} from "../../models/dto/collection-typed.model";
import {QueryParameters} from "../../models/query-parameters/query-parameters.model";
import {NotifierService} from "../../models/services/notifier-service.model";
import {ActionSubActionCompletionsWrapper} from "../../models/stores/base.action";
import {SubResourceUpdateModel} from "../../models/stores/sub-resource-update.model";
import {isNullOrUndefined} from "../../tools";
import {MemoizedUtil} from "../../utils";
import {StoreUtil} from "../../utils/stores/store.util";
import {
  BaseState,
  defaultBaseStateInitValue,
} from "../base/base.state";
import {Relation2TiersActionHelper} from "./relation-2-tiers-action.helper";
import {Relation2TiersNameSpace} from "./relation-2-tiers-namespace.model";
import {Relation2TiersOptions} from "./relation-2-tiers-options.model";
import {Relation2TiersStateModel} from "./relation-2-tiers-state.model";
import {Relation2TiersAction} from "./relation-2-tiers.action";

export const defaultRelation2TiersStateInitValue: () => Relation2TiersStateModel<BaseResourceType> = () =>
  ({
    ...defaultBaseStateInitValue(),
    total: 0,
    selected: undefined,
    current: undefined,
    queryParameters: new QueryParameters(),
  });

// @dynamic
export abstract class Relation2TiersState<TStateModel extends BaseStateModel, TResource extends BaseResourceType, TRelation extends BaseRelationResourceType> extends BaseState<TStateModel> {
  protected readonly _resourceName: ApiResourceNameEnum;
  protected readonly _nameSpace: Relation2TiersNameSpace;

  protected constructor(protected apiService: ApiService,
                        protected store: Store,
                        protected notifierService: NotifierService,
                        protected actions$: Actions,
                        protected options: Relation2TiersOptions) {
    super(apiService, store, notifierService, actions$, options, Relation2TiersState);
    this._resourceName = this.options.resourceName;
  }

  private evaluateSubResourceUrl(parentId: string): string {
    return this._urlResource + urlSeparator + parentId + urlSeparator + this._resourceName;
  }

  static total<TStateModel extends Relation2TiersStateModel<TResource>, TResource extends BaseResourceType, TRelation extends BaseRelationResourceType>(store: Store, ctor: Type<Relation2TiersState<TStateModel, TResource, TRelation>>): Observable<number> {
    return MemoizedUtil.select(store, ctor, state => state.total, true);
  }

  static totalSnapshot<TStateModel extends Relation2TiersStateModel<TResource>, TResource extends BaseResourceType, TRelation extends BaseRelationResourceType>(store: Store, ctor: Type<Relation2TiersState<TStateModel, TResource, TRelation>>): number {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.total);
  }

  static current<TStateModel extends Relation2TiersStateModel<TResource>, TResource extends BaseResourceType, TRelation extends BaseRelationResourceType>(store: Store, ctor: Type<Relation2TiersState<TStateModel, TResource, TRelation>>): Observable<TResource> {
    return MemoizedUtil.select(store, ctor, state => state.current, true);
  }

  static currentSnapshot<TStateModel extends Relation2TiersStateModel<TResource>, TResource extends BaseResourceType, TRelation extends BaseRelationResourceType>(store: Store, ctor: Type<Relation2TiersState<TStateModel, TResource, TRelation>>): TResource {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.current);
  }

  static selected<TStateModel extends Relation2TiersStateModel<TResource>, TResource extends BaseResourceType, TRelation extends BaseRelationResourceType>(store: Store, ctor: Type<Relation2TiersState<TStateModel, TResource, TRelation>>): Observable<TResource[]> {
    return MemoizedUtil.select(store, ctor, state => state.selected, true);
  }

  static selectedSnapshot<TStateModel extends Relation2TiersStateModel<TResource>, TResource extends BaseResourceType, TRelation extends BaseRelationResourceType>(store: Store, ctor: Type<Relation2TiersState<TStateModel, TResource, TRelation>>): TResource[] {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.selected);
  }

  static queryParameters<TStateModel extends Relation2TiersStateModel<TResource>, TResource extends BaseResourceType, TRelation extends BaseRelationResourceType>(store: Store, ctor: Type<Relation2TiersState<TStateModel, TResource, TRelation>>): Observable<QueryParameters> {
    return MemoizedUtil.select(store, ctor, state => state.queryParameters, true);
  }

  static queryParametersSnapshot<TStateModel extends Relation2TiersStateModel<TResource>, TResource extends BaseResourceType, TRelation extends BaseRelationResourceType>(store: Store, ctor: Type<Relation2TiersState<TStateModel, TResource, TRelation>>): QueryParameters {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.queryParameters);
  }

  @RegisterDefaultAction((relation2TiersNameSpace: Relation2TiersNameSpace) => relation2TiersNameSpace.GetAll)
  getAll<U>(ctx: StateContext<Relation2TiersStateModel<TResource>>, action: Relation2TiersAction.GetAll): Observable<CollectionTyped<U>> {
    let reset = {};
    if (!action.keepCurrentContext) {
      reset = {
        total: 0,
        selected: undefined,
      };
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      queryParameters: StoreUtil.getQueryParametersToApply(action.queryParameters, ctx),
      ...reset,
    });
    const url = this.evaluateSubResourceUrl(action.parentId);
    return this.apiService.get<U>(url, ctx.getState().queryParameters)
      .pipe(
        tap((collection: CollectionTyped<U>) => {
          ctx.dispatch(Relation2TiersActionHelper.getAllSuccess<U>(this._nameSpace, action, collection));
        }),
        catchError(error => {
          ctx.dispatch(Relation2TiersActionHelper.getAllFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((relation2TiersNameSpace: Relation2TiersNameSpace) => relation2TiersNameSpace.GetAllSuccess)
  getAllSuccess(ctx: StateContext<Relation2TiersStateModel<TResource>>, action: Relation2TiersAction.GetAllSuccess<TResource>): void {
    ctx.patchState({
      selected: action.list._data,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((relation2TiersNameSpace: Relation2TiersNameSpace) => relation2TiersNameSpace.GetAllFail)
  getAllFail(ctx: StateContext<Relation2TiersStateModel<TResource>>, action: Relation2TiersAction.GetAllFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((relation2TiersNameSpace: Relation2TiersNameSpace) => relation2TiersNameSpace.GetById)
  getById<U>(ctx: StateContext<Relation2TiersStateModel<TResource>>, action: Relation2TiersAction.GetById): Observable<U> {
    let reset = {};
    if (!action.keepCurrentContext) {
      reset = {
        current: undefined,
      };
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      ...reset,
    });
    const url = this.evaluateSubResourceUrl(action.parentId);
    return this.apiService.getById<U>(url, action.resId)
      .pipe(
        tap((model: U) => {
          ctx.dispatch(Relation2TiersActionHelper.getByIdSuccess<U>(this._nameSpace, action, model));
        }),
        catchError(error => {
          ctx.dispatch(Relation2TiersActionHelper.getByIdFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((relation2TiersNameSpace: Relation2TiersNameSpace) => relation2TiersNameSpace.GetByIdSuccess)
  getByIdSuccess(ctx: StateContext<Relation2TiersStateModel<TResource>>, action: Relation2TiersAction.GetByIdSuccess<TResource>): void {
    ctx.patchState({
      current: action.model,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((relation2TiersNameSpace: Relation2TiersNameSpace) => relation2TiersNameSpace.GetByIdFail)
  getByIdFail(ctx: StateContext<Relation2TiersStateModel<TResource>>, action: Relation2TiersAction.GetByIdFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((relation2TiersNameSpace: Relation2TiersNameSpace) => relation2TiersNameSpace.Create)
  create(ctx: StateContext<Relation2TiersStateModel<TResource>>, action: Relation2TiersAction.Create): Observable<string[]> {
    if (action.listResId.length === 0) {
      ctx.dispatch(Relation2TiersActionHelper.createSuccess(this._nameSpace, action));
      return;
    }
    const url = this.evaluateSubResourceUrl(action.parentId);
    return this.apiService.post<string[]>(url, action.listResId)
      .pipe(
        tap(() => {
          ctx.dispatch(Relation2TiersActionHelper.createSuccess(this._nameSpace, action));
        }),
        catchError(error => {
          ctx.dispatch(Relation2TiersActionHelper.createFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((relation2TiersNameSpace: Relation2TiersNameSpace) => relation2TiersNameSpace.DeleteList)
  deleteList(ctx: StateContext<Relation2TiersStateModel<TResource>>, action: Relation2TiersAction.DeleteList): Observable<string[]> {
    if (action.listResId.length === 0) {
      ctx.dispatch(Relation2TiersActionHelper.deleteListSuccess(this._nameSpace, action));
      return;
    }
    const url = this.evaluateSubResourceUrl(action.parentId);
    return this.apiService.delete<string[]>(url, action.listResId)
      .pipe(
        tap(() => {
          ctx.dispatch(Relation2TiersActionHelper.deleteListSuccess(this._nameSpace, action));
        }),
        catchError(error => {
          ctx.dispatch(Relation2TiersActionHelper.deleteListFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((relation2TiersNameSpace: Relation2TiersNameSpace) => relation2TiersNameSpace.Delete)
  delete(ctx: StateContext<Relation2TiersStateModel<TResource>>, action: Relation2TiersAction.Delete): Observable<number> {
    const url = this.evaluateSubResourceUrl(action.parentId);
    return this.apiService.delete<number>(url + urlSeparator + action.resId)
      .pipe(
        tap(() => {
          ctx.dispatch(Relation2TiersActionHelper.deleteSuccess(this._nameSpace, action));
        }),
        catchError(error => {
          ctx.dispatch(Relation2TiersActionHelper.deleteFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((relation2TiersNameSpace: Relation2TiersNameSpace) => relation2TiersNameSpace.UpdateRelation)
  updateRelation(ctx: StateContext<Relation2TiersStateModel<TResource>>, action: Relation2TiersAction.UpdateRelation<TRelation>): Observable<TRelation> {
    const url = this.evaluateSubResourceUrl(action.parentId);
    return this.apiService.patchById<TRelation>(url, action.resId, action.relation)
      .pipe(
        tap(() => {
          ctx.dispatch(Relation2TiersActionHelper.updateRelationSuccess<TRelation>(this._nameSpace, action));
        }),
        catchError(error => {
          ctx.dispatch(Relation2TiersActionHelper.updateRelationFail<TRelation>(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((relation2TiersNameSpace: Relation2TiersNameSpace) => relation2TiersNameSpace.Update)
  update(ctx: StateContext<Relation2TiersStateModel<TResource>>, action: Relation2TiersAction.Update): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    this.internalUpdate(ctx, action).subscribe(success => {
      if (success) {
        ctx.dispatch(Relation2TiersActionHelper.updateSuccess(this._nameSpace, action, action.parentId));
      } else {
        ctx.dispatch(Relation2TiersActionHelper.updateFail(this._nameSpace, action, action.parentId));
      }
    });
  }

  protected internalUpdate(ctx: StateContext<Relation2TiersStateModel<TResource>>, action: Relation2TiersAction.Update): Observable<boolean> {
    const listOldResId = isNullOrUndefined(ctx.getState().selected) ? [] : ctx.getState().selected.map(s => s.resId);
    const listNewResId = action.newResId;

    const subResourceUpdate: SubResourceUpdateModel = StoreUtil.determineSubResourceChange(listOldResId, listNewResId);
    return StoreUtil.dispatchSequentialActionAndWaitForSubActionsCompletion(
      ctx,
      this.computeUpdatesActionsToDispatch(subResourceUpdate, action.parentId),
    );
  }

  @RegisterDefaultAction((relation2TiersNameSpace: Relation2TiersNameSpace) => relation2TiersNameSpace.UpdateSuccess)
  updateSuccess(ctx: StateContext<Relation2TiersStateModel<TResource>>, action: Relation2TiersAction.UpdateSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      selected: undefined,
    });
  }

  @RegisterDefaultAction((relation2TiersNameSpace: Relation2TiersNameSpace) => relation2TiersNameSpace.UpdateFail)
  updateFail(ctx: StateContext<Relation2TiersStateModel<TResource>>, action: Relation2TiersAction.UpdateFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    ctx.dispatch(Relation2TiersActionHelper.getAll(this._nameSpace, action.parentId));
  }

  private computeUpdatesActionsToDispatch(subResourceUpdate: SubResourceUpdateModel, parentId: string): ActionSubActionCompletionsWrapper<Relation2TiersAction.Create | Relation2TiersAction.DeleteList>[] {
    return [
      {
        action: Relation2TiersActionHelper.deleteList(this._nameSpace, parentId, subResourceUpdate.resourceToRemoved),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(this._nameSpace.DeleteListSuccess)),
          this.actions$.pipe(ofActionCompleted(this._nameSpace.DeleteListFail)),
        ],
      },
      {
        action: Relation2TiersActionHelper.create(this._nameSpace, parentId, subResourceUpdate.resourceToAdd),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(this._nameSpace.CreateSuccess)),
          this.actions$.pipe(ofActionCompleted(this._nameSpace.CreateFail)),
        ],
      },
    ];
  }

}
