import {Type} from "@angular/core";
import {
  Actions,
  StateContext,
  Store,
} from "@ngxs/store";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {RegisterDefaultAction} from "../../decorators/store.decorator";
import {ModelAttributeEnum} from "../../enums/model-attribute.enum";
import {ApiResourceNameEnum} from "../../enums/partial/api-resource-name.enum";
import {urlSeparator} from "../../enums/partial/routes.enum";
import {SolidifyStateError} from "../../errors";
import {ApiService} from "../../http/api.service";
import {
  BaseResourceType,
  BaseStateModel,
} from "../../models";
import {CollectionTyped} from "../../models/dto/collection-typed.model";
import {QueryParameters} from "../../models/query-parameters/query-parameters.model";
import {NotifierService} from "../../models/services/notifier-service.model";
import {isTrue} from "../../tools";
import {MemoizedUtil} from "../../utils";
import {StoreUtil} from "../../utils/stores/store.util";
import {
  BaseState,
  defaultBaseStateInitValue,
} from "../base/base.state";
import {CompositionActionHelper} from "./composition-action.helper";
import {CompositionNameSpace} from "./composition-namespace.model";
import {CompositionOptions} from "./composition-options.model";
import {CompositionStateModel} from "./composition-state.model";
import {CompositionAction} from "./composition.action";

export const defaultCompositionStateInitValue: () => CompositionStateModel<BaseResourceType> = () =>
  ({
    ...defaultBaseStateInitValue(),
    total: 0,
    list: undefined,
    current: undefined,
    queryParameters: new QueryParameters(),
  });

// @dynamic
export abstract class CompositionState<TStateModel extends BaseStateModel, TResource extends BaseResourceType> extends BaseState<TStateModel> {
  protected readonly _resourceName: ApiResourceNameEnum;
  protected readonly _nameSpace: CompositionNameSpace;

  protected constructor(protected apiService: ApiService,
                        protected store: Store,
                        protected notifierService: NotifierService,
                        protected actions$: Actions,
                        protected options: CompositionOptions) {
    super(apiService, store, notifierService, actions$, options, CompositionState);
    this._resourceName = this.options.resourceName;
  }

  static list<TStateModel extends CompositionStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<CompositionState<TStateModel, TResource>>): Observable<TResource[]> {
    return MemoizedUtil.select(store, ctor, state => state.list, true);
  }

  static listSnapshot<TStateModel extends CompositionStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<CompositionState<TStateModel, TResource>>): TResource[] {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.list);
  }

  static current<TStateModel extends CompositionStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<CompositionState<TStateModel, TResource>>): Observable<TResource> {
    return MemoizedUtil.select(store, ctor, state => state.current, true);
  }

  static currentSnapshot<TStateModel extends CompositionStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<CompositionState<TStateModel, TResource>>): TResource {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.current);
  }

  static total<TStateModel extends CompositionStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<CompositionState<TStateModel, TResource>>): Observable<number> {
    return MemoizedUtil.select(store, ctor, state => state.total, true);
  }

  static totalSnapshot<TStateModel extends CompositionStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<CompositionState<TStateModel, TResource>>): number {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.total);
  }

  static queryParameters<TStateModel extends CompositionStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<CompositionState<TStateModel, TResource>>): Observable<QueryParameters> {
    return MemoizedUtil.select(store, ctor, state => state.queryParameters, true);
  }

  static queryParametersSnapshot<TStateModel extends CompositionStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<CompositionState<TStateModel, TResource>>): QueryParameters {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.queryParameters);
  }

  private evaluateSubResourceUrl(parentId: string): string {
    return this._urlResource + urlSeparator + parentId + urlSeparator + this._resourceName;
  }

  @RegisterDefaultAction((compositionNameSpace: CompositionNameSpace) => compositionNameSpace.ChangeQueryParameters)
  changeQueryParameters(ctx: StateContext<CompositionStateModel<TResource>>, action: CompositionAction.ChangeQueryParameters): void {
    ctx.patchState({
      queryParameters: action.queryParameters,
    });
    if (isTrue(action.getAllAfterChange)) {
      ctx.dispatch(CompositionActionHelper.getAll(this._nameSpace, action.parentId, undefined, action.keepCurrentContext));
    }
  }

  @RegisterDefaultAction((compositionNameSpace: CompositionNameSpace) => compositionNameSpace.GetAll)
  getAll<U>(ctx: StateContext<CompositionStateModel<TResource>>, action: CompositionAction.GetAll): Observable<CollectionTyped<U>> {
    let reset = {};
    if (!action.keepCurrentContext) {
      reset = {
        total: 0,
        list: undefined,
      };
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      queryParameters: StoreUtil.getQueryParametersToApply(action.queryParameters, ctx),
      ...reset,
    });
    const url = this.evaluateSubResourceUrl(action.parentId);
    return this.apiService.get<U>(url, ctx.getState().queryParameters)
      .pipe(
        tap((collection: CollectionTyped<U>) => {
          ctx.dispatch(CompositionActionHelper.getAllSuccess<U>(this._nameSpace, action, collection));
        }),
        catchError(error => {
          ctx.dispatch(CompositionActionHelper.getAllFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((compositionNameSpace: CompositionNameSpace) => compositionNameSpace.GetAllSuccess)
  getAllSuccess(ctx: StateContext<CompositionStateModel<TResource>>, action: CompositionAction.GetAllSuccess<TResource>): void {
    const queryParameters = StoreUtil.updateQueryParameters(ctx, action.list);

    ctx.patchState({
      list: action.list._data,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      queryParameters: queryParameters,
    });
  }

  @RegisterDefaultAction((compositionNameSpace: CompositionNameSpace) => compositionNameSpace.GetAllFail)
  getAllFail(ctx: StateContext<CompositionStateModel<TResource>>, action: CompositionAction.GetAllFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((compositionNameSpace: CompositionNameSpace) => compositionNameSpace.GetById)
  getById<U>(ctx: StateContext<CompositionStateModel<TResource>>, action: CompositionAction.GetById): Observable<U> {
    let reset = {};
    if (!action.keepCurrentContext) {
      reset = {
        current: undefined,
      };
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      ...reset,
    });
    const url = this.evaluateSubResourceUrl(action.parentId);
    return this.apiService.getById<U>(url, action.resId)
      .pipe(
        tap((model: U) => {
          ctx.dispatch(CompositionActionHelper.getByIdSuccess<U>(this._nameSpace, action, model));
        }),
        catchError(error => {
          ctx.dispatch(CompositionActionHelper.getByIdFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((compositionNameSpace: CompositionNameSpace) => compositionNameSpace.GetByIdSuccess)
  getByIdSuccess(ctx: StateContext<CompositionStateModel<TResource>>, action: CompositionAction.GetByIdSuccess<TResource>): void {
    ctx.patchState({
      current: action.model,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((compositionNameSpace: CompositionNameSpace) => compositionNameSpace.GetByIdFail)
  getByIdFail(ctx: StateContext<CompositionStateModel<TResource>>, action: CompositionAction.GetByIdFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((compositionNameSpace: CompositionNameSpace) => compositionNameSpace.Create)
  create(ctx: StateContext<CompositionStateModel<TResource>>, action: CompositionAction.Create<TResource>): Observable<TResource> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });
    const url = this.evaluateSubResourceUrl(action.parentId);
    return this.apiService.post<TResource>(url, action.model)
      .pipe(
        tap(() => {
          ctx.dispatch(CompositionActionHelper.createSuccess(this._nameSpace, action, action.parentId));
        }),
        catchError(error => {
          ctx.dispatch(CompositionActionHelper.createFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((compositionNameSpace: CompositionNameSpace) => compositionNameSpace.CreateSuccess)
  createSuccess(ctx: StateContext<CompositionStateModel<TResource>>, action: CompositionAction.CreateSuccess<TResource>): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    ctx.dispatch(CompositionActionHelper.getAll(this._nameSpace, action.parentId));
  }

  @RegisterDefaultAction((compositionNameSpace: CompositionNameSpace) => compositionNameSpace.CreateFail)
  createFail(ctx: StateContext<CompositionStateModel<TResource>>, action: CompositionAction.CreateFail<TResource>): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((compositionNameSpace: CompositionNameSpace) => compositionNameSpace.Delete)
  delete(ctx: StateContext<CompositionStateModel<TResource>>, action: CompositionAction.Delete<TResource>): Observable<TResource> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });
    const url = this.evaluateSubResourceUrl(action.parentId);
    return this.apiService.deleteById<TResource>(url, action.model[ModelAttributeEnum.resId])
      .pipe(
        tap(() => {
          ctx.dispatch(CompositionActionHelper.deleteSuccess(this._nameSpace, action, action.parentId));
        }),
        catchError(error => {
          ctx.dispatch(CompositionActionHelper.deleteFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((compositionNameSpace: CompositionNameSpace) => compositionNameSpace.DeleteSuccess)
  deleteSuccess(ctx: StateContext<CompositionStateModel<TResource>>, action: CompositionAction.DeleteSuccess<TResource>): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    ctx.dispatch(CompositionActionHelper.getAll(this._nameSpace, action.parentId));
  }

  @RegisterDefaultAction((compositionNameSpace: CompositionNameSpace) => compositionNameSpace.DeleteFail)
  deleteFail(ctx: StateContext<CompositionStateModel<TResource>>, action: CompositionAction.DeleteFail<TResource>): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((compositionNameSpace: CompositionNameSpace) => compositionNameSpace.Update)
  update(ctx: StateContext<CompositionStateModel<TResource>>, action: CompositionAction.Update<TResource>): Observable<TResource> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });
    const url = this.evaluateSubResourceUrl(action.parentId);
    return this.apiService.patchById<TResource>(url, action.model[ModelAttributeEnum.resId], action.model)
      .pipe(
        tap(() => {
          ctx.dispatch(CompositionActionHelper.updateSuccess(this._nameSpace, action, action.parentId));
        }),
        catchError(error => {
          ctx.dispatch(CompositionActionHelper.updateFail(this._nameSpace, action, action.parentId));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((compositionNameSpace: CompositionNameSpace) => compositionNameSpace.UpdateSuccess)
  updateSuccess(ctx: StateContext<CompositionStateModel<TResource>>, action: CompositionAction.UpdateSuccess<TResource>): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      list: undefined,
    });
  }

  @RegisterDefaultAction((compositionNameSpace: CompositionNameSpace) => compositionNameSpace.UpdateFail)
  updateFail(ctx: StateContext<CompositionStateModel<TResource>>, action: CompositionAction.UpdateFail<TResource>): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    ctx.dispatch(CompositionActionHelper.getAll(this._nameSpace, action.parentId));
  }
}
