import {Type} from "@angular/core";
import {
  Actions,
  ofActionCompleted,
  StateContext,
  Store,
} from "@ngxs/store";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {RegisterDefaultAction} from "../../decorators/store.decorator";
import {ApiResourceNameEnum} from "../../enums/partial/api-resource-name.enum";
import {urlSeparator} from "../../enums/partial/routes.enum";
import {SolidifyStateError} from "../../errors";
import {ApiService} from "../../http/api.service";
import {
  BaseResourceType,
  BaseStateModel,
} from "../../models";
import {CollectionTyped} from "../../models/dto/collection-typed.model";
import {QueryParameters} from "../../models/query-parameters/query-parameters.model";
import {NotifierService} from "../../models/services/notifier-service.model";
import {ActionSubActionCompletionsWrapper} from "../../models/stores/base.action";
import {SubResourceUpdateModel} from "../../models/stores/sub-resource-update.model";
import {isNullOrUndefined} from "../../tools";
import {MemoizedUtil} from "../../utils";
import {StoreUtil} from "../../utils/stores/store.util";
import {
  BaseState,
  defaultBaseStateInitValue,
} from "../base/base.state";
import {AssociationActionHelper} from "./association-action.helper";
import {AssociationNameSpace} from "./association-namespace.model";
import {AssociationOptions} from "./association-options.model";
import {AssociationStateModel} from "./association-state.model";
import {AssociationAction} from "./association.action";

export const defaultAssociationStateInitValue: () => AssociationStateModel<BaseResourceType> = () =>
  ({
    ...defaultBaseStateInitValue(),
    total: 0,
    selected: undefined,
    current: undefined,
    queryParameters: new QueryParameters(),
  });

// @dynamic
export abstract class AssociationState<TStateModel extends BaseStateModel, TResource extends BaseResourceType> extends BaseState<TStateModel> {
  protected readonly _resourceName: ApiResourceNameEnum;
  protected readonly _nameSpace: AssociationNameSpace;

  protected constructor(protected apiService: ApiService,
                        protected store: Store,
                        protected notifierService: NotifierService,
                        protected actions$: Actions,
                        protected options: AssociationOptions) {
    super(apiService, store, notifierService, actions$, options, AssociationState);
    this._resourceName = this.options.resourceName;
  }

  private evaluateSubResourceUrl(parentId: string): string {
    return this._urlResource + urlSeparator + parentId + urlSeparator + this._resourceName;
  }

  static selected<TStateModel extends AssociationStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<AssociationState<TStateModel, TResource>>): Observable<TResource[]> {
    return MemoizedUtil.select(store, ctor, state => state.selected, true);
  }

  static selectedSnapshot<TStateModel extends AssociationStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<AssociationState<TStateModel, TResource>>): TResource[] {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.selected);
  }

  static total<TStateModel extends AssociationStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<AssociationState<TStateModel, TResource>>): Observable<number> {
    return MemoizedUtil.select(store, ctor, state => state.total, true);
  }

  static totalSnapshot<TStateModel extends AssociationStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<AssociationState<TStateModel, TResource>>): number {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.total);
  }

  static current<TStateModel extends AssociationStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<AssociationState<TStateModel, TResource>>): Observable<TResource> {
    return MemoizedUtil.select(store, ctor, state => state.current, true);
  }

  static currentSnapshot<TStateModel extends AssociationStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<AssociationState<TStateModel, TResource>>): TResource {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.current);
  }

  static queryParameters<TStateModel extends AssociationStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<AssociationState<TStateModel, TResource>>): Observable<QueryParameters> {
    return MemoizedUtil.select(store, ctor, state => state.queryParameters, true);
  }

  static queryParametersSnapshot<TStateModel extends AssociationStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<AssociationState<TStateModel, TResource>>): QueryParameters {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.queryParameters);
  }

  @RegisterDefaultAction((associationNameSpace: AssociationNameSpace) => associationNameSpace.GetAll)
  getAll<U>(ctx: StateContext<AssociationStateModel<TResource>>, action: AssociationAction.GetAll): Observable<CollectionTyped<U>> {
    let reset = {};
    if (!action.keepCurrentContext) {
      reset = {
        total: 0,
        selected: undefined,
      };
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      queryParameters: StoreUtil.getQueryParametersToApply(action.queryParameters, ctx),
      ...reset,
    });
    const url = this.evaluateSubResourceUrl(action.parentId);
    return this.apiService.get<U>(url, ctx.getState().queryParameters)
      .pipe(
        tap((collection: CollectionTyped<U>) => {
          ctx.dispatch(AssociationActionHelper.getAllSuccess<U>(this._nameSpace, action, collection));
        }),
        catchError(error => {
          ctx.dispatch(AssociationActionHelper.getAllFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((associationNameSpace: AssociationNameSpace) => associationNameSpace.GetAllSuccess)
  getAllSuccess(ctx: StateContext<AssociationStateModel<TResource>>, action: AssociationAction.GetAllSuccess<TResource>): void {
    const queryParameters = StoreUtil.updateQueryParameters(ctx, action.list);

    ctx.patchState({
      selected: action.list._data,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      queryParameters: queryParameters,
    });
  }

  @RegisterDefaultAction((associationNameSpace: AssociationNameSpace) => associationNameSpace.GetAllFail)
  getAllFail(ctx: StateContext<AssociationStateModel<TResource>>, action: AssociationAction.GetAllFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((associationNameSpace: AssociationNameSpace) => associationNameSpace.GetById)
  getById<U>(ctx: StateContext<AssociationStateModel<TResource>>, action: AssociationAction.GetById): Observable<U> {
    let reset = {};
    if (!action.keepCurrentContext) {
      reset = {
        current: undefined,
      };
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      ...reset,
    });
    const url = this.evaluateSubResourceUrl(action.parentId);
    return this.apiService.getById<U>(url, action.resId)
      .pipe(
        tap((model: U) => {
          ctx.dispatch(AssociationActionHelper.getByIdSuccess<U>(this._nameSpace, action, model));
        }),
        catchError(error => {
          ctx.dispatch(AssociationActionHelper.getByIdFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((associationNameSpace: AssociationNameSpace) => associationNameSpace.GetByIdSuccess)
  getByIdSuccess(ctx: StateContext<AssociationStateModel<TResource>>, action: AssociationAction.GetByIdSuccess<TResource>): void {
    ctx.patchState({
      current: action.model,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((associationNameSpace: AssociationNameSpace) => associationNameSpace.GetByIdFail)
  getByIdFail(ctx: StateContext<AssociationStateModel<TResource>>, action: AssociationAction.GetByIdFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((associationNameSpace: AssociationNameSpace) => associationNameSpace.Create)
  create(ctx: StateContext<AssociationStateModel<TResource>>, action: AssociationAction.Create): Observable<string[]> {
    if (action.listResId.length === 0) {
      ctx.dispatch(AssociationActionHelper.createSuccess(this._nameSpace, action));
      return;
    }
    const url = this.evaluateSubResourceUrl(action.parentId);
    return this.apiService.post<string[]>(url, action.listResId)
      .pipe(
        tap(() => {
          ctx.dispatch(AssociationActionHelper.createSuccess(this._nameSpace, action));
        }),
        catchError(error => {
          ctx.dispatch(AssociationActionHelper.createFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((associationNameSpace: AssociationNameSpace) => associationNameSpace.Delete)
  delete(ctx: StateContext<AssociationStateModel<TResource>>, action: AssociationAction.Delete): Observable<string[]> {
    if (isNullOrUndefined(action.resId)) {
      ctx.dispatch(AssociationActionHelper.deleteSuccess(this._nameSpace, action));
      return;
    }
    const url = this.evaluateSubResourceUrl(action.parentId);
    return this.apiService.delete<string[]>(url + urlSeparator + action.resId)
      .pipe(
        tap(() => {
          ctx.dispatch(AssociationActionHelper.deleteSuccess(this._nameSpace, action));
        }),
        catchError(error => {
          ctx.dispatch(AssociationActionHelper.deleteFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((associationNameSpace: AssociationNameSpace) => associationNameSpace.DeleteList)
  deleteList(ctx: StateContext<AssociationStateModel<TResource>>, action: AssociationAction.DeleteList): Observable<string[]> {
    if (action.listResId.length === 0) {
      ctx.dispatch(AssociationActionHelper.deleteListSuccess(this._nameSpace, action));
      return;
    }
    const url = this.evaluateSubResourceUrl(action.parentId);
    return this.apiService.delete<string[]>(url, action.listResId)
      .pipe(
        tap(() => {
          ctx.dispatch(AssociationActionHelper.deleteListSuccess(this._nameSpace, action));
        }),
        catchError(error => {
          ctx.dispatch(AssociationActionHelper.deleteListFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((associationNameSpace: AssociationNameSpace) => associationNameSpace.Update)
  update(ctx: StateContext<AssociationStateModel<TResource>>, action: AssociationAction.Update): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    const listOldResId = isNullOrUndefined(ctx.getState().selected) ? [] : ctx.getState().selected.map(s => s.resId);
    const listNewResId = action.newResId;

    const subResourceUpdate: SubResourceUpdateModel = StoreUtil.determineSubResourceChange(listOldResId, listNewResId);
    StoreUtil.dispatchSequentialActionAndWaitForSubActionsCompletion(
      ctx,
      this.computeUpdatesActionsToDispatch(subResourceUpdate, action.parentId),
    ).subscribe(success => {
      if (success) {
        ctx.dispatch(AssociationActionHelper.updateSuccess(this._nameSpace, action, action.parentId));
      } else {
        ctx.dispatch(AssociationActionHelper.updateFail(this._nameSpace, action, action.parentId));
      }
    });
  }

  @RegisterDefaultAction((associationNameSpace: AssociationNameSpace) => associationNameSpace.UpdateSuccess)
  updateSuccess(ctx: StateContext<AssociationStateModel<TResource>>, action: AssociationAction.UpdateSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      selected: undefined,
    });
  }

  @RegisterDefaultAction((associationNameSpace: AssociationNameSpace) => associationNameSpace.UpdateFail)
  updateFail(ctx: StateContext<AssociationStateModel<TResource>>, action: AssociationAction.UpdateFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    ctx.dispatch(AssociationActionHelper.getAll(this._nameSpace, action.parentId));
  }

  private computeUpdatesActionsToDispatch(subResourceUpdate: SubResourceUpdateModel, parentId: string): ActionSubActionCompletionsWrapper<AssociationAction.Create | AssociationAction.Delete>[] {
    return [
      {
        action: AssociationActionHelper.deleteList(this._nameSpace, parentId, subResourceUpdate.resourceToRemoved),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(this._nameSpace.DeleteListSuccess)),
          this.actions$.pipe(ofActionCompleted(this._nameSpace.DeleteListFail)),
        ],
      },
      {
        action: AssociationActionHelper.create(this._nameSpace, parentId, subResourceUpdate.resourceToAdd),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(this._nameSpace.CreateSuccess)),
          this.actions$.pipe(ofActionCompleted(this._nameSpace.CreateFail)),
        ],
      },
    ];
  }
}
