import {Type} from "@angular/core";
import {
  Actions,
  ofActionCompleted,
  StateContext,
  Store,
} from "@ngxs/store";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {RegisterDefaultAction} from "../../decorators/store.decorator";
import {ApiResourceNameEnum} from "../../enums/partial/api-resource-name.enum";
import {urlSeparator} from "../../enums/partial/routes.enum";
import {SolidifyStateError} from "../../errors";
import {ApiService} from "../../http/api.service";
import {
  BaseResourceType,
  BaseStateModel,
} from "../../models";
import {CollectionTyped} from "../../models/dto/collection-typed.model";
import {QueryParameters} from "../../models/query-parameters/query-parameters.model";
import {NotifierService} from "../../models/services/notifier-service.model";
import {ActionSubActionCompletionsWrapper} from "../../models/stores/base.action";
import {SubResourceUpdateModel} from "../../models/stores/sub-resource-update.model";
import {isNullOrUndefined} from "../../tools";
import {MemoizedUtil} from "../../utils";
import {StoreUtil} from "../../utils/stores/store.util";
import {
  BaseState,
  defaultBaseStateInitValue,
} from "../base/base.state";
import {AssociationRemoteActionHelper} from "./association-remote-action.helper";
import {AssociationRemoteNameSpace} from "./association-remote-namespace.model";
import {AssociationRemoteOptions} from "./association-remote-options.model";
import {AssociationRemoteStateModel} from "./association-remote-state.model";
import {AssociationRemoteAction} from "./association-remote.action";

export const defaultAssociationRemoteStateInitValue: () => AssociationRemoteStateModel<BaseResourceType> = () =>
  ({
    ...defaultBaseStateInitValue(),
    total: 0,
    selected: undefined,
    current: undefined,
    queryParameters: new QueryParameters(),
  });

// @dynamic
export abstract class AssociationRemoteState<TStateModel extends BaseStateModel, TResource extends BaseResourceType> extends BaseState<TStateModel> {
  protected readonly _resourceName: ApiResourceNameEnum;
  protected readonly _nameSpace: AssociationRemoteNameSpace;

  protected constructor(protected apiService: ApiService,
                        protected store: Store,
                        protected notifierService: NotifierService,
                        protected actions$: Actions,
                        protected options: AssociationRemoteOptions) {
    super(apiService, store, notifierService, actions$, options, AssociationRemoteState);
    this._resourceName = this.options.resourceName;
  }

  private evaluateSubResourceUrl(parentId: string): string {
    return this._urlResource + urlSeparator + parentId + urlSeparator + this._resourceName;
  }

  static selected<TStateModel extends AssociationRemoteStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<AssociationRemoteState<TStateModel, TResource>>): Observable<TResource[]> {
    return MemoizedUtil.select(store, ctor, state => state.selected, true);
  }

  static selectedSnapshot<TStateModel extends AssociationRemoteStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<AssociationRemoteState<TStateModel, TResource>>): TResource[] {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.selected);
  }

  static total<TStateModel extends AssociationRemoteStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<AssociationRemoteState<TStateModel, TResource>>): Observable<number> {
    return MemoizedUtil.select(store, ctor, state => state.total, true);
  }

  static totalSnapshot<TStateModel extends AssociationRemoteStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<AssociationRemoteState<TStateModel, TResource>>): number {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.total);
  }

  static current<TStateModel extends AssociationRemoteStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<AssociationRemoteState<TStateModel, TResource>>): Observable<TResource> {
    return MemoizedUtil.select(store, ctor, state => state.current, true);
  }

  static currentSnapshot<TStateModel extends AssociationRemoteStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<AssociationRemoteState<TStateModel, TResource>>): TResource {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.current);
  }

  static queryParameters<TStateModel extends AssociationRemoteStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<AssociationRemoteState<TStateModel, TResource>>): Observable<QueryParameters> {
    return MemoizedUtil.select(store, ctor, state => state.queryParameters, true);
  }

  static queryParametersSnapshot<TStateModel extends AssociationRemoteStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<AssociationRemoteState<TStateModel, TResource>>): QueryParameters {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.queryParameters);
  }

  @RegisterDefaultAction((associationRemoteNameSpace: AssociationRemoteNameSpace) => associationRemoteNameSpace.GetAll)
  getAll<U>(ctx: StateContext<AssociationRemoteStateModel<TResource>>, action: AssociationRemoteAction.GetAll): Observable<CollectionTyped<U>> {
    let reset = {};
    if (!action.keepCurrentContext) {
      reset = {
        total: 0,
        selected: undefined,
      };
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      queryParameters: StoreUtil.getQueryParametersToApply(action.queryParameters, ctx),
      ...reset,
    });
    const url = this.evaluateSubResourceUrl(action.parentId);
    return this.apiService.get<U>(url, ctx.getState().queryParameters)
      .pipe(
        tap((collection: CollectionTyped<U>) => {
          ctx.dispatch(AssociationRemoteActionHelper.getAllSuccess<U>(this._nameSpace, action, collection));
        }),
        catchError(error => {
          ctx.dispatch(AssociationRemoteActionHelper.getAllFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((associationRemoteNameSpace: AssociationRemoteNameSpace) => associationRemoteNameSpace.GetAllSuccess)
  getAllSuccess(ctx: StateContext<AssociationRemoteStateModel<TResource>>, action: AssociationRemoteAction.GetAllSuccess<TResource>): void {
    const queryParameters = StoreUtil.updateQueryParameters(ctx, action.list);

    ctx.patchState({
      selected: action.list._data,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      queryParameters: queryParameters,
    });
  }

  @RegisterDefaultAction((associationRemoteNameSpace: AssociationRemoteNameSpace) => associationRemoteNameSpace.GetAllFail)
  getAllFail(ctx: StateContext<AssociationRemoteStateModel<TResource>>, action: AssociationRemoteAction.GetAllFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((associationRemoteNameSpace: AssociationRemoteNameSpace) => associationRemoteNameSpace.GetById)
  getById<U>(ctx: StateContext<AssociationRemoteStateModel<TResource>>, action: AssociationRemoteAction.GetById): Observable<U> {
    let reset = {};
    if (!action.keepCurrentContext) {
      reset = {
        current: undefined,
      };
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      ...reset,
    });
    const url = this.evaluateSubResourceUrl(action.parentId);
    return this.apiService.getById<U>(url, action.resId)
      .pipe(
        tap((model: U) => {
          ctx.dispatch(AssociationRemoteActionHelper.getByIdSuccess<U>(this._nameSpace, action, model));
        }),
        catchError(error => {
          ctx.dispatch(AssociationRemoteActionHelper.getByIdFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((associationRemoteNameSpace: AssociationRemoteNameSpace) => associationRemoteNameSpace.GetByIdSuccess)
  getByIdSuccess(ctx: StateContext<AssociationRemoteStateModel<TResource>>, action: AssociationRemoteAction.GetByIdSuccess<TResource>): void {
    ctx.patchState({
      current: action.model,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((associationRemoteNameSpace: AssociationRemoteNameSpace) => associationRemoteNameSpace.GetByIdFail)
  getByIdFail(ctx: StateContext<AssociationRemoteStateModel<TResource>>, action: AssociationRemoteAction.GetByIdFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((associationRemoteNameSpace: AssociationRemoteNameSpace) => associationRemoteNameSpace.Create)
  create(ctx: StateContext<AssociationRemoteStateModel<TResource>>, action: AssociationRemoteAction.Create): Observable<string[]> {
    if (action.listResId.length === 0) {
      ctx.dispatch(AssociationRemoteActionHelper.createSuccess(this._nameSpace, action));
      return;
    }
    const url = this.evaluateSubResourceUrl(action.parentId);
    return this.apiService.post<string[]>(url, action.listResId)
      .pipe(
        tap(() => {
          ctx.dispatch(AssociationRemoteActionHelper.createSuccess(this._nameSpace, action));
        }),
        catchError(error => {
          ctx.dispatch(AssociationRemoteActionHelper.createFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((associationRemoteNameSpace: AssociationRemoteNameSpace) => associationRemoteNameSpace.Delete)
  delete(ctx: StateContext<AssociationRemoteStateModel<TResource>>, action: AssociationRemoteAction.Delete): Observable<string[]> {
    if (isNullOrUndefined(action.resId)) {
      ctx.dispatch(AssociationRemoteActionHelper.deleteSuccess(this._nameSpace, action));
      return;
    }
    const url = this.evaluateSubResourceUrl(action.parentId);
    return this.apiService.delete<string[]>(url + urlSeparator + action.resId)
      .pipe(
        tap(() => {
          ctx.dispatch(AssociationRemoteActionHelper.deleteSuccess(this._nameSpace, action));
        }),
        catchError(error => {
          ctx.dispatch(AssociationRemoteActionHelper.deleteFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((associationRemoteNameSpace: AssociationRemoteNameSpace) => associationRemoteNameSpace.DeleteList)
  deleteList(ctx: StateContext<AssociationRemoteStateModel<TResource>>, action: AssociationRemoteAction.DeleteList): Observable<string[]> {
    if (action.listResId.length === 0) {
      ctx.dispatch(AssociationRemoteActionHelper.deleteListSuccess(this._nameSpace, action));
      return;
    }
    const url = this.evaluateSubResourceUrl(action.parentId);
    return this.apiService.delete<string[]>(url, action.listResId)
      .pipe(
        tap(() => {
          ctx.dispatch(AssociationRemoteActionHelper.deleteListSuccess(this._nameSpace, action));
        }),
        catchError(error => {
          ctx.dispatch(AssociationRemoteActionHelper.deleteListFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((associationRemoteNameSpace: AssociationRemoteNameSpace) => associationRemoteNameSpace.Update)
  update(ctx: StateContext<AssociationRemoteStateModel<TResource>>, action: AssociationRemoteAction.Update): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    const listOldResId = isNullOrUndefined(ctx.getState().selected) ? [] : ctx.getState().selected.map(s => s.resId);
    const listNewResId = action.newResId;

    const subResourceUpdate: SubResourceUpdateModel = StoreUtil.determineSubResourceChange(listOldResId, listNewResId);
    StoreUtil.dispatchSequentialActionAndWaitForSubActionsCompletion(
      ctx,
      this.computeUpdatesActionsToDispatch(subResourceUpdate, action.parentId),
    ).subscribe(success => {
      if (success) {
        ctx.dispatch(AssociationRemoteActionHelper.updateSuccess(this._nameSpace, action, action.parentId));
      } else {
        ctx.dispatch(AssociationRemoteActionHelper.updateFail(this._nameSpace, action, action.parentId));
      }
    });
  }

  @RegisterDefaultAction((associationRemoteNameSpace: AssociationRemoteNameSpace) => associationRemoteNameSpace.UpdateSuccess)
  updateSuccess(ctx: StateContext<AssociationRemoteStateModel<TResource>>, action: AssociationRemoteAction.UpdateSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      selected: undefined,
    });
  }

  @RegisterDefaultAction((associationRemoteNameSpace: AssociationRemoteNameSpace) => associationRemoteNameSpace.UpdateFail)
  updateFail(ctx: StateContext<AssociationRemoteStateModel<TResource>>, action: AssociationRemoteAction.UpdateFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    ctx.dispatch(AssociationRemoteActionHelper.getAll(this._nameSpace, action.parentId));
  }

  private computeUpdatesActionsToDispatch(subResourceUpdate: SubResourceUpdateModel, parentId: string): ActionSubActionCompletionsWrapper<AssociationRemoteAction.Create | AssociationRemoteAction.Delete>[] {
    return [
      {
        action: AssociationRemoteActionHelper.deleteList(this._nameSpace, parentId, subResourceUpdate.resourceToRemoved),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(this._nameSpace.DeleteListSuccess)),
          this.actions$.pipe(ofActionCompleted(this._nameSpace.DeleteListFail)),
        ],
      },
      {
        action: AssociationRemoteActionHelper.create(this._nameSpace, parentId, subResourceUpdate.resourceToAdd),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(this._nameSpace.CreateSuccess)),
          this.actions$.pipe(ofActionCompleted(this._nameSpace.CreateFail)),
        ],
      },
    ];
  }
}
