import {Type} from "@angular/core";
import {
  Actions,
  StateContext,
  Store,
} from "@ngxs/store";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {RegisterDefaultAction} from "../../decorators/store.decorator";
import {ApiResourceNameEnum} from "../../enums/partial/api-resource-name.enum";
import {urlSeparator} from "../../enums/partial/routes.enum";
import {SolidifyStateError} from "../../errors";
import {ApiService} from "../../http/api.service";
import {
  BaseResourceType,
  BaseStateModel,
} from "../../models";
import {CollectionTyped} from "../../models/dto/collection-typed.model";
import {QueryParameters} from "../../models/query-parameters/query-parameters.model";
import {NotifierService} from "../../models/services/notifier-service.model";
import {MemoizedUtil} from "../../utils";
import {StoreUtil} from "../../utils/stores/store.util";
import {
  BaseState,
  defaultBaseStateInitValue,
} from "../base/base.state";
import {AssociationNoSqlReadOnlyActionHelper} from "./association-no-sql-read-only-action.helper";
import {AssociationNoSqlReadOnlyNameSpace} from "./association-no-sql-read-only-namespace.model";
import {AssociationNoSqlReadOnlyOptions} from "./association-no-sql-read-only-options.model";
import {AssociationNoSqlReadOnlyStateModel} from "./association-no-sql-read-only-state.model";
import {AssociationNoSqlReadOnlyAction} from "./association-no-sql-read-only.action";

export const defaultAssociationNoSqlReadOnlyStateInitValue: () => AssociationNoSqlReadOnlyStateModel<BaseResourceType> = () =>
  ({
    ...defaultBaseStateInitValue(),
    total: 0,
    selected: undefined,
    current: undefined,
    queryParameters: new QueryParameters(),
  });

// @dynamic
export abstract class AssociationNoSqlReadOnlyState<TStateModel extends BaseStateModel, TResource extends BaseResourceType> extends BaseState<TStateModel> {
  protected readonly _resourceName: ApiResourceNameEnum;
  protected readonly _nameSpace: AssociationNoSqlReadOnlyNameSpace;

  protected constructor(protected apiService: ApiService,
                        protected store: Store,
                        protected notifierService: NotifierService,
                        protected actions$: Actions,
                        protected options: AssociationNoSqlReadOnlyOptions) {
    super(apiService, store, notifierService, actions$, options, AssociationNoSqlReadOnlyState);
    this._resourceName = this.options.resourceName;
  }

  private evaluateSubResourceUrl(parentId: string): string {
    return this._urlResource + urlSeparator + parentId + urlSeparator + this._resourceName;
  }

  static selected<TStateModel extends AssociationNoSqlReadOnlyStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<AssociationNoSqlReadOnlyState<TStateModel, TResource>>): Observable<TResource[]> {
    return MemoizedUtil.select(store, ctor, state => state.selected, true);
  }

  static selectedSnapshot<TStateModel extends AssociationNoSqlReadOnlyStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<AssociationNoSqlReadOnlyState<TStateModel, TResource>>): TResource[] {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.selected);
  }

  static total<TStateModel extends AssociationNoSqlReadOnlyStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<AssociationNoSqlReadOnlyState<TStateModel, TResource>>): Observable<number> {
    return MemoizedUtil.select(store, ctor, state => state.total, true);
  }

  static totalSnapshot<TStateModel extends AssociationNoSqlReadOnlyStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<AssociationNoSqlReadOnlyState<TStateModel, TResource>>): number {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.total);
  }

  static current<TStateModel extends AssociationNoSqlReadOnlyStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<AssociationNoSqlReadOnlyState<TStateModel, TResource>>): Observable<TResource> {
    return MemoizedUtil.select(store, ctor, state => state.current, true);
  }

  static currentSnapshot<TStateModel extends AssociationNoSqlReadOnlyStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<AssociationNoSqlReadOnlyState<TStateModel, TResource>>): TResource {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.current);
  }

  static queryParameters<TStateModel extends AssociationNoSqlReadOnlyStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<AssociationNoSqlReadOnlyState<TStateModel, TResource>>): Observable<QueryParameters> {
    return MemoizedUtil.select(store, ctor, state => state.queryParameters, true);
  }

  static queryParametersSnapshot<TStateModel extends AssociationNoSqlReadOnlyStateModel<TResource>, TResource extends BaseResourceType>(store: Store, ctor: Type<AssociationNoSqlReadOnlyState<TStateModel, TResource>>): QueryParameters {
    return MemoizedUtil.selectSnapshot(store, ctor, state => state.queryParameters);
  }

  @RegisterDefaultAction((associationNoSqlReadOnlyNameSpace: AssociationNoSqlReadOnlyNameSpace) => associationNoSqlReadOnlyNameSpace.GetAll)
  getAll<U>(ctx: StateContext<AssociationNoSqlReadOnlyStateModel<TResource>>, action: AssociationNoSqlReadOnlyAction.GetAll): Observable<CollectionTyped<U>> {
    let reset = {};
    if (!action.keepCurrentContext) {
      reset = {
        total: 0,
        selected: undefined,
      };
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      queryParameters: StoreUtil.getQueryParametersToApply(action.queryParameters, ctx),
      ...reset,
    });

    const url = this.evaluateSubResourceUrl(action.parentId);
    return this.apiService.get<U>(url, ctx.getState().queryParameters)
      .pipe(
        tap((collection: CollectionTyped<U>) => {
          ctx.dispatch(AssociationNoSqlReadOnlyActionHelper.getAllSuccess<U>(this._nameSpace, action, collection));
        }),
        catchError(error => {
          ctx.dispatch(AssociationNoSqlReadOnlyActionHelper.getAllFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((associationNoSqlReadOnlyNameSpace: AssociationNoSqlReadOnlyNameSpace) => associationNoSqlReadOnlyNameSpace.GetAllSuccess)
  getAllSuccess(ctx: StateContext<AssociationNoSqlReadOnlyStateModel<TResource>>, action: AssociationNoSqlReadOnlyAction.GetAllSuccess<TResource>): void {
    const queryParameters = StoreUtil.updateQueryParameters(ctx, action.list);

    ctx.patchState({
      total: action.list._page.totalItems,
      selected: action.list._data,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      queryParameters,
    });
  }

  @RegisterDefaultAction((associationNoSqlReadOnlyNameSpace: AssociationNoSqlReadOnlyNameSpace) => associationNoSqlReadOnlyNameSpace.GetAllFail)
  getAllFail(ctx: StateContext<AssociationNoSqlReadOnlyStateModel<TResource>>, action: AssociationNoSqlReadOnlyAction.GetAllFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((associationNoSqlReadOnlyNameSpace: AssociationNoSqlReadOnlyNameSpace) => associationNoSqlReadOnlyNameSpace.GetById)
  getById<U>(ctx: StateContext<AssociationNoSqlReadOnlyStateModel<TResource>>, action: AssociationNoSqlReadOnlyAction.GetById): Observable<U> {
    let reset = {};
    if (!action.keepCurrentContext) {
      reset = {
        current: undefined,
      };
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      ...reset,
    });
    const url = this.evaluateSubResourceUrl(action.parentId);
    return this.apiService.getById<U>(url, action.resId)
      .pipe(
        tap((model: U) => {
          ctx.dispatch(AssociationNoSqlReadOnlyActionHelper.getByIdSuccess<U>(this._nameSpace, action, model));
        }),
        catchError(error => {
          ctx.dispatch(AssociationNoSqlReadOnlyActionHelper.getByIdFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((associationNoSqlReadOnlyNameSpace: AssociationNoSqlReadOnlyNameSpace) => associationNoSqlReadOnlyNameSpace.GetByIdSuccess)
  getByIdSuccess(ctx: StateContext<AssociationNoSqlReadOnlyStateModel<TResource>>, action: AssociationNoSqlReadOnlyAction.GetByIdSuccess<TResource>): void {
    ctx.patchState({
      current: action.model,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((associationNoSqlReadOnlyNameSpace: AssociationNoSqlReadOnlyNameSpace) => associationNoSqlReadOnlyNameSpace.GetByIdFail)
  getByIdFail(ctx: StateContext<AssociationNoSqlReadOnlyStateModel<TResource>>, action: AssociationNoSqlReadOnlyAction.GetByIdFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }
}
