import {
  Inject,
  Injectable,
} from "@angular/core";
import {TranslateService} from "@ngx-translate/core";
import {NotificationTypeEnum} from "../enums/notification-type.enum";
import {DefaultSolidifyEnvironment} from "../environments/environment.solidify-defaults";
import {ENVIRONMENT} from "../injection-tokens";
import {NotifierService} from "../models/services/notifier-service.model";
import {
  SNACK_BAR,
  SnackbarData,
  SnackBarModel,
  SnackbarOption,
} from "../models/snackbar/snack-bar.model";
import {isNullOrUndefined} from "../tools";

@Injectable({
  providedIn: "root",
})
export class NotificationService implements NotifierService {
  private readonly ONE_SECOND_IN_MILLISECONDS: number = 1000;

  constructor(@Inject(SNACK_BAR) public snackBar: SnackBarModel,
              private translate: TranslateService,
              @Inject(ENVIRONMENT) private environment: DefaultSolidifyEnvironment) {
  }

  showInformation(message: string, messageParam: Object | undefined = undefined): void {
    this.show({
      durationInSecond: this.environment.defaultNotificationInformationDurationInSeconds,
      data: this.getDataOption(message, messageParam, NotificationTypeEnum.information),
      component: this.environment.defaultNotificationComponent,
    });
  }

  showWarning(message: string, messageParam: Object | undefined = undefined): void {
    this.show({
      durationInSecond: this.environment.defaultNotificationWarningDurationInSeconds,
      data: this.getDataOption(message, messageParam, NotificationTypeEnum.warning),
      component: this.environment.defaultNotificationComponent,
    });
  }

  showSuccess(message: string, messageParam: Object | undefined = undefined): void {
    this.show({
      durationInSecond: this.environment.defaultNotificationSuccessDurationInSeconds,
      data: this.getDataOption(message, messageParam, NotificationTypeEnum.success),
      component: this.environment.defaultNotificationComponent,
    });
  }

  showError(message: string, messageParam: Object | undefined = true): void {
    this.show({
      durationInSecond: this.environment.defaultNotificationErrorDurationInSeconds,
      data: this.getDataOption(message, messageParam, NotificationTypeEnum.error),
      component: this.environment.defaultNotificationComponent,
    });
  }

  private getDataOption(message: string, messageParam: Object | undefined, category: NotificationTypeEnum): SnackbarData {
    return {
      message: message,
      messageParam: messageParam,
      category: category,
    };
  }

  show(option: SnackbarOption): void {
    if (isNullOrUndefined(option.component)) {
      let message = option.data.message;
      const messageParam = option.data.messageParam;
      if (isNullOrUndefined(messageParam)) {
        message = this.translate.instant(message);
      } else {
        message = this.translate.instant(message, messageParam);
      }
      this.snackBar.open(message, "X", {
        panelClass: [option.data.category, "native"],
        duration: option.durationInSecond * this.ONE_SECOND_IN_MILLISECONDS,
      });
    } else {
      this.snackBar.openFromComponent(option.component, {
        data: option.data,
        panelClass: [option.data.category, "component"],
        duration: option.durationInSecond * this.ONE_SECOND_IN_MILLISECONDS,
      });
    }
  }

  dismiss(): void {
    this.snackBar.dismiss();
  }
}
