import {HttpErrorResponse} from "@angular/common/http";
import {
  ErrorHandler,
  Injectable,
  Injector,
} from "@angular/core";
import {DefaultSolidifyEnvironment} from "../environments";
import {SolidifyError} from "../errors";
import {ErrorHelper} from "../helpers/error.helper";
import {ENVIRONMENT} from "../injection-tokens/environment.injection-token";
import {NOTIFIER_SERVICE} from "../injection-tokens/notifier-service.injection-token";
import {NotifierService} from "../models/services/notifier-service.model";
import {
  isInstanceOf,
  isTruthyObject,
  isUndefined,
} from "../tools";
import {ErrorService} from "./error.service";
import {LoggingService} from "./logging.service";

@Injectable({
  providedIn: "root",
})
export class GlobalErrorsHandlerService implements ErrorHandler {

  private _environment: DefaultSolidifyEnvironment | undefined;

  get environment(): DefaultSolidifyEnvironment {
    if (isUndefined(this._environment)) {
      this._environment = this.injector.get(ENVIRONMENT);
    }
    return this._environment;
  }

  private _loggingService: LoggingService | undefined;

  get loggingService(): LoggingService {
    if (isUndefined(this._loggingService)) {
      this._loggingService = this.injector.get(LoggingService);
    }
    return this._loggingService;
  }

  private _errorService: ErrorService | undefined;

  get errorService(): ErrorService {
    if (isUndefined(this._errorService)) {
      this._errorService = this.injector.get(ErrorService);
    }
    return this._errorService;
  }

  private _notifierService: NotifierService | undefined;

  get notifierService(): NotifierService {
    if (isUndefined(this._notifierService)) {
      this._notifierService = this.injector.get(NOTIFIER_SERVICE);
    }
    return this._notifierService;
  }

  // Error handling is important and needs to be loaded first.
  // Because of this we should manually inject the services with Injector.
  constructor(protected injector: Injector) {
  }

  handleError(error: Error | HttpErrorResponse | SolidifyError<Error> | SolidifyError<HttpErrorResponse>): void {
    if (error === this.environment.errorToSkipInErrorHandler) {
      return;
    }
    const httpErrorKeyToSkipInErrorHandler = this.environment.httpErrorKeyToSkipInErrorHandler;
    const notifierService = this.notifierService;
    const nativeError = isInstanceOf(error, SolidifyError) ? error.nativeError : error as Error;

    let message;
    let stackTrace;

    if (nativeError instanceof HttpErrorResponse) {
      if (ErrorHelper.isErrorToTreat(nativeError, httpErrorKeyToSkipInErrorHandler)) {
        // Server Error
        message = this.errorService.getServerMessage(nativeError);
        stackTrace = this.errorService.getServerStack(nativeError);
        if (isTruthyObject(notifierService)) {
          notifierService.showError(message, undefined);
        }
      }
    } else {
      // Client Error
      message = this.errorService.getClientMessage(nativeError);
      stackTrace = this.errorService.getClientStack(nativeError);
      if (isTruthyObject(notifierService)) {
        notifierService.showError(message, undefined);
      }
    }

    // Always log errors
    this.loggingService.logError(message, error, stackTrace);
  }
}
