import {
  Observable,
  Subscription,
} from "rxjs";
import {SubscriptionUtil} from "../../utils/subscription.util";

export class SubscriptionManager {

  private readonly _subscriptions: Subscription[] = [];

  geSubscriptions(): Subscription[] {
    return this._subscriptions.slice();
  }

  subscribe<T>(observable: Observable<T>,
               onNext?: (value: T) => void,
               onError?: (error: Error) => void,
               onComplete?: () => void): Subscription {
    const subscription = observable.subscribe(onNext, onError, onComplete);
    this.add(subscription);
    return subscription;
  }

  add(subscription: Subscription): boolean {
    if (SubscriptionUtil.isUnsubscribable(subscription)) {
      this._subscriptions.push(subscription);
      return true;
    }
    return false;
  }

  addMultiple(...subscriptions: Subscription[]): number {
    return subscriptions.reduce((seed, next) => seed + (this.add(next) ? 1 : 0), 0);
  }

  remove(subscription: Subscription): boolean {
    const index = this._subscriptions.indexOf(subscription);
    if (index < 0) {
      return false;
    }
    this._subscriptions.splice(index, 1);
    return true;
  }

  clear(): boolean {
    if (this._subscriptions.length === 0) {
      return false;
    }
    this._subscriptions.splice(0);
    return true;
  }

  removeAndUnsubscribe(subscription: Subscription): boolean {
    return this.remove(subscription) && SubscriptionUtil.unsubscribe(subscription);
  }

  clearAndUnsubscribeAll(): number {
    return this._subscriptions.splice(0).reduce((seed, next) => seed + (SubscriptionUtil.unsubscribe(next) ? 1 : 0), 0);
  }

}
