import {
  HttpErrorResponse,
  HttpEvent,
  HttpHandler,
  HttpInterceptor,
  HttpRequest,
} from "@angular/common/http";
import {
  Inject,
  Injectable,
} from "@angular/core";
import {
  Observable,
  throwError,
} from "rxjs";
import {
  catchError,
  mergeMap,
} from "rxjs/operators";
import {OAuth2Service} from "../auth/oauth2.service";
import {OAuthStorage} from "../auth/types";
import {DefaultSolidifyEnvironment} from "../environments/environment.solidify-defaults";
import {ENVIRONMENT} from "../injection-tokens";
import {HttpStatus} from "../lib-wrapped/http-status-code.model";
import {isArray} from "../tools/is/is.tool";

@Injectable({
  providedIn: "root",
})
export class OAuth2Interceptor implements HttpInterceptor {

  constructor(private authStorage: OAuthStorage,
              private oAuth2Service: OAuth2Service,
              @Inject(ENVIRONMENT) private environment: DefaultSolidifyEnvironment) {
  }

  private checkUrl(url: string): boolean {
    return isArray(this.environment.allowedUrls) && this.environment.allowedUrls.includes(url);
  }

  public intercept(req: HttpRequest<any>, next: HttpHandler, noRetry: boolean = false): Observable<HttpEvent<any>> {

    const url = req.url.toLowerCase();

    if (this.checkUrl(url) || url === this.environment.tokenEndpoint) {
      return next.handle(req);
    }

    const sendAccessToken = this.environment.sendAccessToken;

    if (sendAccessToken && this.authStorage.getItem("access_token")) {
      const token = this.authStorage.getItem("access_token");
      const header = "Bearer " + token;
      const headers = req.headers.set("Authorization", header);
      req = req.clone({headers});
    }

    return next
      .handle(req)
      .pipe(catchError((err: HttpErrorResponse) => {
        if (!noRetry && (err.status === HttpStatus.UNAUTHORIZED || err.status === HttpStatus.NOT_ACCEPTABLE)) {
          return this.oAuth2Service.refreshToken()
            .pipe(
              mergeMap(value => {
                if (value) {
                  return this.intercept(req, next, true);
                }
                return throwError(err);
              }),
            );
        }
        return throwError(err);
      }));
  }

}
