import {
  HttpClient,
  HttpEvent,
  HttpHeaders,
  HttpParams,
  HttpResponse,
} from "@angular/common/http";
import {
  Injectable,
  Optional,
} from "@angular/core";
import {Observable} from "rxjs";
import {ResourceApiEnum} from "../enums/partial/resource-api.enum";
import {CollectionTyped} from "../models/dto/collection-typed.model";
import {QueryParameters} from "../models/query-parameters/query-parameters.model";
import {MappingObjectUtil} from "../utils/mapping-object.util";
import {QueryParametersUtil} from "../utils/query-parameters.util";
import {Configuration} from "./configuration";
import {CustomHttpUrlEncodingCodec} from "./encoder";

@Injectable({
  providedIn: "root",
})
export class ApiService {
  public defaultHeaders: HttpHeaders = new HttpHeaders();
  public configuration: Configuration = new Configuration();

  constructor(protected httpClient: HttpClient, @Optional() configuration: Configuration) {
    if (configuration) {
      this.configuration = configuration;
    }
  }

  /**
   * GET
   * To get a collection of resource
   * @param path The resource path
   * @param queryParameters The query parameters (page size, current page, sort field)
   * @param customParameters To apply n other query parameters.
   * @param observe set whether or not to return the data Observable as the body, response or events. defaults to returning the body.
   * @param reportProgress flag to report request and response progress.
   */
  public get<T>(path: ResourceApiEnum, queryParameters: QueryParameters, customParameters?: { [key: string]: string; },
                observe?: "body", reportProgress?: boolean): Observable<CollectionTyped<T>>;
  public get<T>(path: ResourceApiEnum, queryParameters: QueryParameters, customParameters?: { [key: string]: string; },
                observe?: "response", reportProgress?: boolean): Observable<HttpResponse<CollectionTyped<T>>>;
  public get<T>(path: ResourceApiEnum, queryParameters: QueryParameters, customParameters?: { [key: string]: string; },
                observe?: "events", reportProgress?: boolean): Observable<HttpEvent<CollectionTyped<T>>>;
  public get<T>(path: ResourceApiEnum, queryParameters?: QueryParameters, customParameters?: { [key: string]: string; },
                observe: any = "body", reportProgress: boolean = false): Observable<any> {
    const queryParametersHttp = this.getQueryParameters(queryParameters, customParameters);
    const headers = this.getHeaders();

    return this.httpClient.get<CollectionTyped<T>>(`${path}`,
      {
        params: queryParametersHttp,
        withCredentials: this.configuration.withCredentials,
        headers,
        observe,
        reportProgress,
      },
    );
  }

  /**
   * GET
   * To get a list of resource
   * @param path The resource path
   * @param queryParameters The query parameters (page size, current page, sort field)
   * @param customParameters To apply n other query parameters.
   * @param observe set whether or not to return the data Observable as the body, response or events. defaults to returning the body.
   * @param reportProgress flag to report request and response progress.
   */
  public getList<T>(path: ResourceApiEnum, queryParameters: QueryParameters, customParameters?: { [key: string]: string; },
                    observe?: "body", reportProgress?: boolean): Observable<T[]>;
  public getList<T>(path: ResourceApiEnum, queryParameters: QueryParameters, customParameters?: { [key: string]: string; },
                    observe?: "response", reportProgress?: boolean): Observable<HttpResponse<T[]>>;
  public getList<T>(path: ResourceApiEnum, queryParameters: QueryParameters, customParameters?: { [key: string]: string; },
                    observe?: "events", reportProgress?: boolean): Observable<HttpEvent<T[]>>;
  public getList<T>(path: ResourceApiEnum, queryParameters?: QueryParameters, customParameters?: { [key: string]: string; },
                    observe: any = "body", reportProgress: boolean = false): Observable<any> {
    const queryParametersHttp = this.getQueryParameters(queryParameters, customParameters);
    const headers = this.getHeaders();

    return this.httpClient.get<T[]>(`${path}`,
      {
        params: queryParametersHttp,
        withCredentials: this.configuration.withCredentials,
        headers,
        observe,
        reportProgress,
      },
    );
  }

  /**
   * GET BY ID
   * To get a specific resource
   * @param path The resource path
   * @param resId Resource Id (GUID)
   * @param observe set whether or not to return the data Observable as the body, response or events. defaults to returning the body.
   * @param reportProgress flag to report request and response progress.
   */
  public getById<T>(path: ResourceApiEnum, resId: string, observe?: "body", reportProgress?: boolean): Observable<T>;
  public getById<T>(path: ResourceApiEnum, resId: string, observe?: "response", reportProgress?: boolean): Observable<HttpResponse<T>>;
  public getById<T>(path: ResourceApiEnum, resId: string, observe?: "events", reportProgress?: boolean): Observable<HttpEvent<T>>;
  public getById<T>(path: ResourceApiEnum, resId: string, observe: any = "body", reportProgress: boolean = false): Observable<any> {
    if (resId === null || resId === undefined) {
      throw new Error(`Required parameter resId was null or undefined when calling ${path}GetById.`);
    }
    const headers = this.getHeaders();

    return this.getByIdInPath<T>(`${path}/${encodeURIComponent(String(resId))}`, observe, reportProgress);
  }

  /**
   * GET BY ID
   * To get a specific resource
   * @param path The resource path
   * @param observe set whether or not to return the data Observable as the body, response or events. defaults to returning the body.
   * @param reportProgress flag to report request and response progress.
   */
  public getByIdInPath<T>(path: ResourceApiEnum, observe?: "body", reportProgress?: boolean): Observable<T>;
  public getByIdInPath<T>(path: ResourceApiEnum, observe?: "response", reportProgress?: boolean): Observable<HttpResponse<T>>;
  public getByIdInPath<T>(path: ResourceApiEnum, observe?: "events", reportProgress?: boolean): Observable<HttpEvent<T>>;
  public getByIdInPath<T>(path: ResourceApiEnum, observe: any = "body", reportProgress: boolean = false): Observable<any> {
    const headers = this.getHeaders();

    return this.httpClient.get<T>(path.toString(),
      {
        withCredentials: this.configuration.withCredentials,
        headers,
        observe,
        reportProgress,
      },
    );
  }

  /**
   * DELETE
   * To delete an existing resource
   * @param path The resource path
   * @param resId Resource Id (GUID)
   * @param observe set whether or not to return the data Observable as the body, response or events. defaults to returning the body.
   * @param reportProgress flag to report request and response progress.
   */
  public deleteById<T>(path: ResourceApiEnum, resId: string, observe?: "body", reportProgress?: boolean): Observable<T>;
  public deleteById<T>(path: ResourceApiEnum, resId: string, observe?: "response", reportProgress?: boolean): Observable<HttpResponse<T>>;
  public deleteById<T>(path: ResourceApiEnum, resId: string, observe?: "events", reportProgress?: boolean): Observable<HttpEvent<T>>;
  public deleteById<T>(path: ResourceApiEnum, resId: string, observe: any = "body", reportProgress: boolean = false): Observable<any> {
    if (resId === null || resId === undefined) {
      throw new Error(`Required parameter resId was null or undefined when calling ${path}DeleteById.`);
    }
    const headers = this.getHeaders();

    return this.httpClient.delete<T>(`${path}/${encodeURIComponent(String(resId))}`,
      {
        withCredentials: this.configuration.withCredentials,
        headers,
        observe,
        reportProgress,
      },
    );
  }

  /**
   * PATCH
   * To edit a existing resource
   * @param path The resource path
   * @param resId Resource Id (GUID)
   * @param body The body to object to update
   * @param observe set whether or not to return the data Observable as the body, response or events. defaults to returning the body.
   * @param reportProgress flag to report request and response progress.
   */
  public patchById<T, U = T>(path: ResourceApiEnum, resId: string, body?: T, observe?: "body", reportProgress?: boolean): Observable<U>;
  public patchById<T, U = T>(path: ResourceApiEnum, resId: string, body?: T,
                             observe?: "response", reportProgress?: boolean): Observable<HttpResponse<U>>;
  public patchById<T, U = T>(path: ResourceApiEnum, resId: string, body?: T,
                             observe?: "events", reportProgress?: boolean): Observable<HttpEvent<U>>;
  public patchById<T, U = T>(path: ResourceApiEnum, resId: string, body?: T,
                             observe: any = "body", reportProgress: boolean = false): Observable<any> {
    if (resId === null || resId === undefined) {
      throw new Error("Required parameter resId was null or undefined when calling PatchById method.");
    }
    let headers = this.getHeaders();
    headers = this.addContentType(headers);

    return this.httpClient.patch<T>(`${path}/${encodeURIComponent(String(resId))}`,
      body,
      {
        withCredentials: this.configuration.withCredentials,
        headers,
        observe,
        reportProgress,
      },
    ) as Observable<HttpEvent<U>>;
  }

  /**
   * PATCH
   * To edit a existing resource
   * @param full path The resource path containing ResourceId
   * @param body The body to object to update
   * @param observe set whether or not to return the data Observable as the body, response or events. defaults to returning the body.
   * @param reportProgress flag to report request and response progress.
   */
  public patchByIdWithCustomUrl<T, U = T>(path: ResourceApiEnum, body?: T, observe?: "body", reportProgress?: boolean): Observable<U>;
  public patchByIdWithCustomUrl<T, U = T>(path: ResourceApiEnum, body?: T, observe?: "response", reportProgress?: boolean): Observable<HttpResponse<U>>;
  public patchByIdWithCustomUrl<T, U = T>(path: ResourceApiEnum, body?: T, observe?: "events", reportProgress?: boolean): Observable<HttpEvent<U>>;
  public patchByIdWithCustomUrl<T, U = T>(path: ResourceApiEnum, body?: T, observe: any = "body", reportProgress: boolean = false): Observable<any> {
    let headers = this.getHeaders();
    headers = this.addContentType(headers);

    return this.httpClient.patch<T>(`${path}`,
      body,
      {
        withCredentials: this.configuration.withCredentials,
        headers,
        observe,
        reportProgress,
      },
    ) as Observable<HttpEvent<U>>;
  }

  /**
   * POST
   * To create a new resource
   * @param path The resource path
   * @param body The body of object to create
   * @param observe set whether or not to return the data Observable as the body, response or events. defaults to returning the body.
   * @param reportProgress flag to report request and response progress.
   */
  public post<T, U = T>(path: ResourceApiEnum, body?: T, observe?: "body", reportProgress?: boolean): Observable<U>;
  public post<T, U = T>(path: ResourceApiEnum, body?: T, observe?: "response", reportProgress?: boolean): Observable<HttpResponse<U>>;
  public post<T, U = T>(path: ResourceApiEnum, body?: T, observe?: "events", reportProgress?: boolean): Observable<HttpEvent<U>>;
  public post<T, U = T>(path: ResourceApiEnum, body?: T, observe: any = "body", reportProgress: boolean = false): Observable<any> {
    let headers = this.getHeaders();
    headers = this.addContentType(headers);

    return this.httpClient.post<T>(`${path}`,
      body,
      {
        withCredentials: this.configuration.withCredentials,
        headers,
        observe,
        reportProgress,
      },
    ) as Observable<HttpEvent<U>>;
  }

  public delete<T, U = T>(path: string, body?: T, observe?: "body", reportProgress?: boolean): Observable<U>;
  public delete<T, U = T>(path: string, body?: T, observe?: "response", reportProgress?: boolean): Observable<HttpResponse<U>>;
  public delete<T, U = T>(path: string, body?: T, observe?: "events", reportProgress?: boolean): Observable<HttpEvent<U>>;
  public delete<T, U = T>(path: string, body?: T, observe: any = "body", reportProgress: boolean = false): Observable<any> {
    let headers = this.getHeaders();
    headers = this.addContentType(headers);

    return this.httpClient.delete<T>(`${path}`,
      {
        withCredentials: this.configuration.withCredentials,
        headers,
        observe,
        reportProgress,
        body,
      } as any,
    ) as Observable<HttpEvent<U>>;
  }

  public upload<T>(path: string, data: FormData): Observable<HttpEvent<T>> {
    const headers = this.defaultHeaders;
    return this.httpClient.post<any>(`${path}`, data, {
      reportProgress: true,
      observe: "events",
      headers,
    });
  }

  private getQueryParameters(queryParameters: QueryParameters, customParameters: { [p: string]: string }): HttpParams {
    let queryParametersHttp = new HttpParams({encoder: new CustomHttpUrlEncodingCodec()});
    if (queryParameters == null) {
      return queryParametersHttp;
    }

    const size = QueryParametersUtil.getPageSize(queryParameters);
    if (size !== undefined && size !== null) {
      queryParametersHttp = queryParametersHttp.set("size", size as any);
    }

    const page = QueryParametersUtil.getPageIndex(queryParameters);
    if (page !== undefined && page !== null) {
      queryParametersHttp = queryParametersHttp.set("page", page as any);
    }

    const sort = QueryParametersUtil.getSortString(queryParameters);
    if (sort !== undefined && sort !== null) {
      queryParametersHttp = queryParametersHttp.set("sort", sort as any);
    }

    const searchFilters = queryParameters.search;
    if (searchFilters !== undefined && searchFilters !== null) {
      MappingObjectUtil.forEach(searchFilters.searchItems, (value: string, field: string) => {
        queryParametersHttp = queryParametersHttp.set(field, value as any);
      });
    }

    if (customParameters !== undefined && customParameters !== null) {
      // tslint:disable-next-line:forin
      for (const key in customParameters) {
        queryParametersHttp = queryParametersHttp.set(key, customParameters[key] as any);
      }
    }

    return queryParametersHttp;
  }

  private getHeaders(): HttpHeaders {
    let headers = this.defaultHeaders;

    // to determine the Accept header
    const httpHeaderAccepts: string[] = [
      "application/json",
    ];
    const httpHeaderAcceptSelected: string | undefined = this.configuration.selectHeaderAccept(httpHeaderAccepts);
    if (httpHeaderAcceptSelected !== undefined) {
      headers = headers.set("Accept", httpHeaderAcceptSelected);
    }
    return headers;
  }

  private addContentType(headers: HttpHeaders): HttpHeaders {
    const consumes: string[] = [
      "application/json",
    ];
    const httpContentTypeSelected: string | undefined = this.configuration.selectHeaderContentType(consumes);
    if (httpContentTypeSelected !== undefined) {
      headers = headers.set("Content-Type", httpContentTypeSelected);
    }
    return headers;
  }
}
