import {
  AbstractControl,
  FormGroup,
} from "@angular/forms";
import {isNullOrUndefined} from "../tools";
import {StringUtil} from "../utils";

export class FormValidationHelper {
  static readonly METADATA_ERRORS_MESSAGES: string = "ERRORS_MESSAGES";

  static getMetadataErrors(formControl: AbstractControl): Map<string, string> {
    return formControl[FormValidationHelper.METADATA_ERRORS_MESSAGES];
  }

  static initMetadataErrors(formControl: AbstractControl): Map<string, string> {
    return formControl[FormValidationHelper.METADATA_ERRORS_MESSAGES] = new Map<string, string>();
  }

  static getFormControl(form: FormGroup, key: string): AbstractControl | undefined {
    return form.get(key);
  }

  static getFormErrorByName(form: FormGroup, formDefinition: string): string {
    const formControl = this.getFormControl(form, formDefinition);
    return this.getFormError(formControl);
  }

  static getFormError(formControl: AbstractControl): string {
    if (isNullOrUndefined(formControl)) {
      return StringUtil.stringEmpty;
    }
    const errors = FormValidationHelper.getMetadataErrors(formControl);
    if (isNullOrUndefined(errors) || errors.size === 0) {
      return StringUtil.stringEmpty;
    }
    return Array.from(errors.values()).join(". ");
  }

  static hasRequiredFieldByName(form: FormGroup, abstractControlName: string): boolean {
    const formControl = this.getFormControl(form, abstractControlName);
    return this.hasRequiredField(formControl);
  }

  static hasRequiredField(abstractControl: AbstractControl): boolean {
    if (abstractControl.validator) {
      const validator = abstractControl.validator({} as AbstractControl);
      if (validator && validator.required) {
        return true;
      }
    }
    if (abstractControl["controls"]) {
      for (const controlName in abstractControl["controls"]) {
        if (abstractControl["controls"][controlName]) {
          if (this.hasRequiredField(abstractControl["controls"][controlName])) {
            return true;
          }
        }
      }
    }
    return false;
  }
}
