import {
  ChangeDetectorRef,
  Directive,
  ElementRef,
  Injector,
  Input,
  OnInit,
  Renderer2,
  Self,
} from "@angular/core";
import {
  NgControl,
  ValidationErrors,
} from "@angular/forms";
import {TranslateService} from "@ngx-translate/core";
import {
  filter,
  tap,
} from "rxjs/operators";
import {CoreAbstractDirective} from "../core-abstract/core-abstract.directive";
import {DefaultSolidifyEnvironment} from "../../environments";
import {FormValidationHelper} from "../../helpers";
import {ENVIRONMENT} from "../../injection-tokens";
import {
  isArray,
  isEmptyArray,
  isNullOrUndefined,
  isUndefined,
} from "../../tools";
import {ObjectUtil} from "../../utils";

@Directive({
  selector: "[solidifyValidation]",
})
export class SolidifyValidationDirective extends CoreAbstractDirective implements OnInit {
  private _previousErrors: ValidationErrors;

  private readonly _ERROR_REQUIRED: string = "required";
  private readonly _ERROR_BACKEND: string = "errorsFromBackend";

  private readonly _REGISTRY_INVALID_KEY: string = "INVALID";

  @Input("solidifyValidation")
  matError: any | undefined;

  private _environment: DefaultSolidifyEnvironment | undefined;

  get environment(): DefaultSolidifyEnvironment {
    if (isUndefined(this._environment)) {
      this._environment = this._injector.get(ENVIRONMENT);
    }
    return this._environment;
  }

  constructor(private _injector: Injector,
              @Self() private _ngControl: NgControl,
              private _elementRef: ElementRef,
              private _renderer: Renderer2,
              private _translate: TranslateService,
              private _changeDetector: ChangeDetectorRef) {
    super();
  }

  private addError(errorKey: string, errorMessage: string): void {
    this.getMetadataErrors().set(errorKey, errorMessage);
  }

  private getMetadataErrors(): Map<string, string> {
    return FormValidationHelper.getMetadataErrors(this._ngControl.control);
  }

  ngOnInit(): void {
    FormValidationHelper.initMetadataErrors(this._ngControl.control);
    this.updateErrorsMetadata(this._ngControl.control.status as FormState);
    this.subscribe(this._ngControl.control.statusChanges.pipe(
      filter(s => this.isDifferent()),
      tap(validity => {
        this.updateErrorsMetadata(validity);
      })),
    );
  }

  private updateErrorsMetadata(validity: FormState): void {
    const errors = this._ngControl.control.errors;
    switch (validity) {
      case FormState.valid:
        this.cleanExistingErrors();
        break;
      case FormState.invalid:
        this.computeNewErrors(errors);
        break;
    }
    this._previousErrors = ObjectUtil.clone(errors);
  }

  private cleanExistingErrors(): void {
    this.getMetadataErrors().clear();
  }

  private isDifferent(): boolean {
    let newErrorsObject = this._ngControl.control.errors;
    let oldErrorsObject = this._previousErrors;

    if (isNullOrUndefined(newErrorsObject)) {
      newErrorsObject = {};
    }
    if (isNullOrUndefined(oldErrorsObject)) {
      oldErrorsObject = {};
    }

    if (Object.getOwnPropertyNames(newErrorsObject).length !== Object.getOwnPropertyNames(oldErrorsObject).length) {
      return true;
    }

    for (const property of Object.getOwnPropertyNames(newErrorsObject)) {
      if (oldErrorsObject.hasOwnProperty(property) && typeof oldErrorsObject[property] === typeof newErrorsObject[property]) {

        if (isArray(oldErrorsObject[property])) {
          if (oldErrorsObject[property].length !== newErrorsObject[property].length) {
            return true;
          }
          for (let i = 0; i < oldErrorsObject[property].length; i++) {
            if (oldErrorsObject[property][i] !== newErrorsObject[property][i]) {
              return true;
            }
          }
        } else {
          if (oldErrorsObject[property] !== newErrorsObject[property]) {
            return true;
          }
        }
      }
    }

    return false;
  }

  private computeNewErrors(errors: ValidationErrors): void {
    let errorFound = this.getMetadataErrors().size > 0;
    if (isNullOrUndefined(errors)) {
      throw Error("Should never be here");
    }

    if (errors[this._ERROR_REQUIRED] === true && !this.getMetadataErrors().has(this._ERROR_REQUIRED)) {
      errorFound = true;
      this.addError(this._ERROR_REQUIRED, this._translate.instant(this.environment.validationErrorRequiredToTranslate));
    }

    if (!isNullOrUndefined(errors[this._ERROR_BACKEND]) && !isEmptyArray(errors[this._ERROR_BACKEND])) {
      for (let i = 0; i < errors[this._ERROR_BACKEND].length; i++) {
        const key = this._ERROR_BACKEND + errors[this._ERROR_BACKEND][i];
        if (!this.getMetadataErrors().has(key)) {
          errorFound = true;
          this.addError(key, this._translate.instant(errors[this._ERROR_BACKEND][i]));
        }
      }
    }

    if (!errorFound) {
      this.addError(this._REGISTRY_INVALID_KEY, this._translate.instant(this.environment.validationErrorInvalidToTranslate));
    }

    if (this.matError) {
      this.matError.textContent = FormValidationHelper.getFormError(this._ngControl.control);
    }

    this._changeDetector.detectChanges();
  }
}

enum FormState {
  valid = "VALID",
  invalid = "INVALID",
}
