import {
  HttpClientTestingModule,
  HttpTestingController,
} from "@angular/common/http/testing";
import {TestBed} from "@angular/core/testing";
import {DefaultSolidifyEnvironment} from "../environments/environment.solidify-defaults";
import {ENVIRONMENT} from "../injection-tokens";

import {AppConfigService} from "./app-config.service";

describe("AppConfigService", () => {
  let httpMock: HttpTestingController;
  let service: AppConfigService;
  const environment = {} as DefaultSolidifyEnvironment;
  const CONFIG_FILE_PATH = "./assets/configurations/environment.runtime.json";

  beforeEach(() => {
    TestBed.configureTestingModule({
      imports: [HttpClientTestingModule],
      providers: [
        {
          provide: ENVIRONMENT,
          useValue: environment,
        },
      ],
    });
    httpMock = TestBed.get(HttpTestingController);
    service = TestBed.get(AppConfigService);
  });

  it("should be created", () => {
    expect(service).toBeTruthy();
  });

  it("should override compile environment config with runtime configuration", done => {
    const compiledEnvironmentConfig: any = {production: true, envConfig: "initial"};
    const runtimeEnvironmentConfig: any = {production: true, envConfig: "updated"};

    service.mergeConfig(compiledEnvironmentConfig).subscribe(() => {
      expect(compiledEnvironmentConfig).toEqual(runtimeEnvironmentConfig);
      done();
    });
    const request = httpMock.expectOne(CONFIG_FILE_PATH);
    request.flush(runtimeEnvironmentConfig);
  });

  it("should add additional runtime configuration to the compile environment config", done => {
    const compiledEnvironmentConfig: any = {production: true, envConfig: "a_value"};
    const runtimeEnvironmentConfig: any = {newConfig: "a_value"};
    const expectedMergedConfig: any = {production: true, envConfig: "a_value", newConfig: "a_value"};

    service.mergeConfig(compiledEnvironmentConfig).subscribe(() => {
      expect(compiledEnvironmentConfig).toEqual(expectedMergedConfig);
      done();
    });
    const request = httpMock.expectOne(CONFIG_FILE_PATH);
    request.flush(runtimeEnvironmentConfig);
  });

  it("should throw a console.error when environment runtime config is not a valid JSON format", done => {
    const compiledEnvironmentConfig: any = {production: true, envConfig: "initial"};

    spyOn(console, "error");

    service.mergeConfig(compiledEnvironmentConfig).subscribe(() => {
      expect(console.error).toHaveBeenCalled();
      done();
    });

    httpMock.expectOne(CONFIG_FILE_PATH).error(new ErrorEvent("error"));
  });

  it("should not load runtime environment config if the compile environment is dev", done => {
    const compiledEnvironmentConfig: any = {production: false, envConfig: "initial"};

    spyOn(console, "error");

    service.mergeConfig(compiledEnvironmentConfig).subscribe(() => {
      expect(console.error).not.toHaveBeenCalled();
      done();
    });
  });

  it("should keep compiled environment config when runtime environment config is empty", done => {
    const compiledEnvironmentConfig: any = {production: true, envConfig: "initial"};
    const runtimeEnvironmentConfig = null;

    spyOn(console, "error");

    service.mergeConfig(compiledEnvironmentConfig).subscribe(() => {
      expect(console.error).not.toHaveBeenCalled();
      done();
    });

    const request = httpMock.expectOne(CONFIG_FILE_PATH);
    request.flush(runtimeEnvironmentConfig);
  });
});
