import {
  HttpClient,
  HttpHeaders,
  HttpParams,
} from "@angular/common/http";
import {
  Inject,
  Injectable,
  NgZone,
  Optional,
} from "@angular/core";
import {
  Observable,
  of,
  Subject,
  Subscription,
} from "rxjs";
import {
  delay,
  filter,
  map,
  switchMap,
  tap,
} from "rxjs/operators";
import {DefaultSolidifyEnvironment} from "../environments/environment.solidify-defaults";
import {ENVIRONMENT} from "../injection-tokens";
import {
  OAuthErrorEvent,
  OAuthEvent,
  OAuthInfoEvent,
  OAuthSuccessEvent,
} from "./events";
import {
  LoginOptions,
  OAuthStorage,
  TokenResponse,
} from "./types";

/**
 * Service for logging in and logging out with
 * OIDC and OAuth2. Supports code flow.
 */
@Injectable({
  providedIn: "root",
})
export class OAuth2Service {

  private eventsSubject: Subject<OAuthEvent> = new Subject<OAuthEvent>();
  public events: Observable<OAuthEvent> = this.eventsSubject.asObservable();
  private _storage: OAuthStorage;
  private accessTokenTimeoutSubscription: Subscription;
  private readonly RESPONSE_TYPE: string = "code";
  private readonly timeoutFactor: number = 0.75;

  constructor(private ngZone: NgZone,
              private http: HttpClient,
              @Inject(ENVIRONMENT) private environment: DefaultSolidifyEnvironment,
              @Optional() storage: OAuthStorage) {

    this.events = this.eventsSubject.asObservable();

    try {
      if (storage) {
        this.setStorage(storage);
      } else if (typeof sessionStorage !== "undefined") {
        this.setStorage(sessionStorage);
      }
    } catch (e) {
      console.error("cannot access sessionStorage. " +
        "Consider setting an own storage implementation using setStorage", e);
    }
    this.setupRefreshTimer();
  }

  /**
   * Sets a custom storage used to store the received
   * tokens on client side. By default, the browser's
   * sessionStorage is used.
   *
   * @param storage storage service
   */
  public setStorage(storage: OAuthStorage): void {
    this._storage = storage;
  }

  /**
   * Refreshes the token using a refresh_token.
   * This does not work for implicit flow, b/c
   * there is no refresh_token in this flow.
   */
  public refreshToken(): Observable<boolean> {
    let params = new HttpParams()
      .set("grant_type", "refresh_token")
      .set("refresh_token", this._storage.getItem("refresh_token"))
      .set("scope", this.environment.scope);
    if (this.environment.dummyClientSecret) {
      params = params.set("client_secret", this.environment.dummyClientSecret);
    }
    return this.fetchToken(params);
  }

  /**
   * Setup an automatic refresh token call when the access token expires
   */
  public setupAutomaticRefreshToken(): void {
    this.events.pipe(filter(e => e.type === "token_expires"))
      .pipe(switchMap(() => this.refreshToken()))
      .subscribe();

    this.restartRefreshTimerIfStillLoggedIn();
  }

  /**
   * Starts the authorization code flow and redirects to user to
   * the auth servers login url.
   */
  public initAuthorizationCodeFlow(): void {

    if (!this.validateUrlForHttps(this.environment.loginUrl)) {
      throw new Error("loginUrl must use Http. Also check property requireHttps.");
    }
    this.createLoginUrl("", "")
      .subscribe(url => {
        location.href = url;
      }, error => {
        console.error("Error in initAuthorizationCodeFlow");
        console.error(error);
      });
  }

  /**
   * Checks whether there are tokens in the hash fragment
   * as a result of the implicit flow. These tokens are
   * parsed, validated and used to sign the user in to the
   * current client.
   *
   * @param options Optional options.
   */
  public tryLogin(options: LoginOptions = null): Observable<boolean> {
    if (!this.environment.requestAccessToken && !this.environment.oidc) {
      console.warn("Either requestAccessToken or oidc or both must be true.");
      return of(false);
    }

    if (this.getAccessToken()) {
      return of(true);
    } else if (new Date(this.getAccessTokenExpiration()) >= new Date()) {
      return of(true);
    } else if (window.location.search && (window.location.search.startsWith("?code=") || window.location.search.includes("&code="))) {
      return this.extractCodeAndGetTokenFromCode(window.location.search);
    } else if (window.location.hash && (window.location.hash.includes("?code=") || window.location.hash.includes("&code="))) {
      return this.extractCodeAndGetTokenFromCode(window.location.hash);
    }
    return of(false);
  }

  private extractCodeAndGetTokenFromCode(path: string): Observable<boolean> {
    const parameter = path.split("?")[1].split("&");
    const codeParam = parameter.filter(param => param.includes("code="));
    const code = codeParam.length ? codeParam[0].split("code=")[1] : undefined;
    if (code) {
      try {
        return this.getTokenFromCode(code);
      } catch (e) {
        return of(false);
      }
    }
  }

  /**
   * Returns the received claims about the user.
   */
  public getIdentityClaims(): any {
    const claims = this._storage.getItem("id_token_claims_obj");
    if (!claims) {
      return null;
    }
    return JSON.parse(claims);
  }

  /**
   * Returns the current access_token.
   */
  public getAccessToken(): string {
    return this._storage.getItem("access_token");
  }

  /**
   * Returns the current refresh_token
   */
  public getRefreshToken(): string {
    return this._storage.getItem("refresh_token");
  }

  /**
   * Returns the expiration date of the access_token
   * as milliseconds since 1970.
   */
  public getAccessTokenExpiration(): number {
    if (!this._storage.getItem("expires_at")) {
      return null;
    }
    return parseInt(this._storage.getItem("expires_at"), 10);
  }

  /**
   * Checkes, whether there is a valid access_token.
   */
  public hasValidAccessToken(): boolean {
    if (this.getAccessToken()) {
      const expiresAt = this._storage.getItem("expires_at");
      const now = new Date();

      return !(expiresAt && parseInt(expiresAt, 10) < now.getTime());
    }
    return false;
  }

  /**
   * Removes all tokens and logs the user out.
   * If a logout url is configured, the user is
   * redirected to it.
   * @param noRedirectToLogoutUrl boolean to redirect or not
   */
  public logOut(noRedirectToLogoutUrl: boolean = false): void {
    this.clearStorage();

    this.eventsSubject.next(new OAuthInfoEvent("logout"));

    if (!this.environment.logoutUrl) {
      return;
    }
    if (noRedirectToLogoutUrl) {
      return;
    }
    let logoutUrl: string;

    if (!this.validateUrlForHttps(this.environment.logoutUrl)) {
      throw new Error("logoutUrl must use Http. Also check property requireHttps.");
    }

    // For backward compatibility
    if (this.environment.logoutUrl.indexOf("{{") > -1) {
      logoutUrl = this.environment.logoutUrl
        .replace(/\{\{client_id\}\}/, this.environment.clientId);
    } else {
      logoutUrl =
        this.environment.logoutUrl +
        (this.environment.logoutUrl.indexOf("?") > -1 ? "&" : "?") +
        "id_token_hint=" +
        encodeURIComponent("") +
        "&post_logout_redirect_uri=" +
        encodeURIComponent(this.environment.postLogoutRedirectUri || this.environment.redirectUrl);
    }
    location.href = logoutUrl;
  }

  /**
   * @ignore
   */
  public createAndSaveNonce(): string {
    const nonce = this.createNonce();
    this._storage.setItem("nonce", nonce);
    return nonce;
  }

  protected createNonce(): string {
    let text = "";
    const possible = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789";

    for (let i = 0; i < 40; i++) {
      text += possible.charAt(Math.floor(Math.random() * possible.length));
    }

    return text;
  }

  private restartRefreshTimerIfStillLoggedIn(): void {
    this.setupExpirationTimers();
  }

  private createLoginUrl(state: string = "",
                         customRedirectUri: string = ""): Observable<string> {

    let redirectUri: string;

    if (customRedirectUri) {
      redirectUri = customRedirectUri;
    } else {
      redirectUri = this.environment.redirectUrl;
    }

    let nonce = null;
    if (!this.environment.disableNonceCheck) {
      nonce = this.createAndSaveNonce();
      if (state) {
        state = nonce + this.environment.nonceStateSeparator + state;
      } else {
        state = nonce;
      }
    }

    if (!this.environment.requestAccessToken && !this.environment.oidc) {
      throw new Error("Either requestAccessToken or oidc or both must be true");
    }

    const separationChar = this.environment.logoutUrl.indexOf("?") > -1 ? "&" : "?";

    const scope = this.environment.logoutUrl;

    let url =
      this.environment.loginUrl +
      separationChar +
      "response_type=" +
      encodeURIComponent(this.RESPONSE_TYPE) +
      "&client_id=" +
      encodeURIComponent(this.environment.clientId) +
      "&state=" +
      encodeURIComponent(state) +
      "&redirect_uri=" +
      encodeURIComponent(redirectUri) +
      "&scope=" +
      encodeURIComponent(scope);

    if (nonce && this.environment.oidc) {
      url += "&nonce=" + encodeURIComponent(nonce);
    }

    return of(url);
  }

  private storeAccessTokenResponse(accessToken: string, refreshToken: string, expiresIn: number, grantedScopes: string): void {
    this._storage.setItem("access_token", accessToken);
    if (grantedScopes) {
      this._storage.setItem("granted_scopes", JSON.stringify(grantedScopes.split("+")));
    }
    this._storage.setItem("access_token_stored_at", "" + Date.now());
    if (expiresIn) {
      const expiresInMilliSeconds = expiresIn * 1000;
      const now = new Date();
      const expiresAt = now.getTime() + expiresInMilliSeconds;
      this._storage.setItem("expires_at", "" + expiresAt);
    }

    if (refreshToken) {
      this._storage.setItem("refresh_token", refreshToken);
    }
  }

  private validateUrlForHttps(url: string): boolean {
    if (!url) {
      return true;
    }

    const lcUrl = url.toLowerCase();

    if (this.environment.requireHttps === false) {
      return true;
    }
    return lcUrl.startsWith("https://");
  }

  private setupRefreshTimer(): void {
    if (typeof window === "undefined") {
      console.warn("timer not supported on this plattform");
      return;
    }

    this.events.pipe(filter(e => e.type === "token_received"))
      .subscribe(() => {
        this.clearAccessTokenTimer();
        this.setupExpirationTimers();
      });
  }

  private setupExpirationTimers(): void {
    if (this.hasValidAccessToken()) {
      this.setupAccessTokenTimer();
    }
  }

  private setupAccessTokenTimer(): void {
    const expiration = this.getAccessTokenExpiration();
    const storedAt = this.getAccessTokenStoredAt();
    const timeout = this.calcTimeout(storedAt, expiration);

    this.ngZone.runOutsideAngular(() => {
      this.accessTokenTimeoutSubscription = of(new OAuthInfoEvent("token_expires", "access_token"))
        .pipe(delay(timeout))
        .subscribe(e => {
          this.ngZone.run(() => {
            this.eventsSubject.next(e);
          });
        });
    });
  }

  private clearAccessTokenTimer(): void {
    if (this.accessTokenTimeoutSubscription) {
      this.accessTokenTimeoutSubscription.unsubscribe();
    }
  }

  private calcTimeout(storedAt: number, expiration: number): number {
    return (expiration - storedAt) * this.timeoutFactor;
  }

  /**
   * Get token using an intermediate code. Works for the Authorization Code flow.
   */
  private getTokenFromCode(code: string): Observable<boolean> {
    const params = new HttpParams()
      .set("grant_type", "authorization_code")
      .set("code", code)
      .set("redirect_uri", this.environment.redirectUrl);
    return this.fetchToken(params).pipe(
      tap((success) => this.cleanOAuthInfosInUrl()),
    );
  }

  private cleanOAuthInfosInUrl(): void {
    if (window.history.replaceState) {
      // Prevents browser from storing in history the query param "code" (prevent that usage of back browser navigation cause new authentification)
      window.history.replaceState({}, null, window.location.href.split("?")[0]);
    }
  }

  private fetchToken(params: HttpParams): Observable<boolean> {
    if (!this.validateUrlForHttps(this.environment.tokenEndpoint)) {
      throw new Error("tokenEndpoint must use Http. Also check property requireHttps.");
    }

    params = params.set("client_id", this.environment.clientId);

    const authData = window.btoa(this.environment.clientId + ":" + this.environment.dummyClientSecret);
    const headers = new HttpHeaders()
      .set("Content-Type", "application/x-www-form-urlencoded")
      .set("Authorization", "Basic " + authData);

    return this.http.post<TokenResponse>(this.environment.tokenEndpoint, params, {headers})
      .pipe(
        map((tokenResp) => {
            this.storeAccessTokenResponse(tokenResp.access_token, tokenResp.refresh_token, tokenResp.expires_in, tokenResp.scope);
            this.eventsSubject.next(new OAuthSuccessEvent("token_received"));
            this.eventsSubject.next(new OAuthSuccessEvent("token_refreshed"));
            return true;
          },
          (err) => {
            console.error("Error getting token", err);
            this.eventsSubject.next(new OAuthErrorEvent("token_refresh_error", err));
            throw err;
          },
        ),
      );
  }

  private getAccessTokenStoredAt(): number {
    return parseInt(this._storage.getItem("access_token_stored_at"), 10);
  }

  private clearStorage(): void {
    this._storage.removeItem("access_token");
    this._storage.removeItem("id_token");
    this._storage.removeItem("refresh_token");
    this._storage.removeItem("nonce");
    this._storage.removeItem("expires_at");
    this._storage.removeItem("id_token_claims_obj");
    this._storage.removeItem("id_token_expires_at");
    this._storage.removeItem("id_token_stored_at");
    this._storage.removeItem("access_token_stored_at");
  }

}
