import {HttpClient} from "@angular/common/http";
import {Injectable} from "@angular/core";
import {AdminResourceApiEnum} from "@app/shared/enums/api.enum";
import {LocalStateEnum} from "@app/shared/enums/local-state.enum";
import {AppPersonAction} from "@app/stores/person/app-person.action";
import {
  AppUserAction,
  appUserActionNameSpace,
} from "@app/stores/user/app-user.action";
import {environment} from "@environments/environment";
import {User} from "@models";
import {
  Action,
  Actions,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {ApiActionEnum} from "@shared/enums/api-action.enum";
import {AppTourService} from "@shared/services/app-tour.service";
import {ResourceLogoStateModel} from "@shared/stores/resource-logo/resource-logo-state.model";
import {
  defaultResourceLogoStateInitValue,
  ResourceLogoState,
  ResourceLogoStateModeEnum,
} from "@shared/stores/resource-logo/resource-logo.state";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  CollectionTyped,
  isNullOrUndefined,
  NotificationService,
  QueryParameters,
  SolidifyStateError,
  urlSeparator,
} from "solidify-frontend";

export interface AppUserStateModel extends ResourceLogoStateModel<User> {
}

@Injectable()
@State<AppUserStateModel>({
  name: LocalStateEnum.application_user,
  defaults: {
    ...defaultResourceLogoStateInitValue(),
    queryParameters: new QueryParameters(environment.defaultEnumValuePageSizeOption),
  },
})
export class AppUserState extends ResourceLogoState<AppUserStateModel, User> {
  constructor(protected readonly apiService: ApiService,
              protected readonly store: Store,
              protected readonly notificationService: NotificationService,
              protected readonly actions$: Actions,
              protected readonly _httpClient: HttpClient,
              private readonly _appTourService: AppTourService) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: appUserActionNameSpace,
    }, _httpClient, ResourceLogoStateModeEnum.avatar);
  }

  protected get _urlResource(): string {
    return AdminResourceApiEnum.users;
  }

  protected get _urlLogoResource(): string {
    return AdminResourceApiEnum.people;
  }

  @Action(AppUserAction.GetCurrentUser)
  getCurrentUser(ctx: StateContext<AppUserStateModel>, action: AppUserAction.GetCurrentUser): Observable<CollectionTyped<User>> {
    return this.apiService.get<User>(AdminResourceApiEnum.users + urlSeparator + ApiActionEnum.AUTHENTICATED, null)
      .pipe(
        tap((user: User) => {
          ctx.dispatch(new AppUserAction.GetCurrentUserSuccess(action, user));
        }),
        catchError(error => {
          ctx.dispatch(new AppUserAction.GetCurrentUserFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(AppUserAction.GetCurrentUserSuccess)
  getCurrentUserSuccess(ctx: StateContext<AppUserStateModel>, action: AppUserAction.GetCurrentUserSuccess): void {
    ctx.patchState({
      current: action.user,
    });

    ctx.dispatch(new AppPersonAction.AddCurrentPerson(action.user.person));

    if (isNullOrUndefined(action.user.avatar)) {
      ctx.dispatch(new AppUserAction.GetPhoto(action.user.person.resId));
    }

    if (isNullOrUndefined(action.user.lastLoginTime)) {
      this._appTourService.runFullTour();
    }
  }
}
