import {
  HttpClient,
  HttpHeaders,
  HttpResponse,
} from "@angular/common/http";
import {Injectable} from "@angular/core";
import {BaseResourceApiEnum} from "@app/shared/enums/api.enum";
import {LocalStateEnum} from "@app/shared/enums/local-state.enum";
import {AppTocAction} from "@app/stores/toc/app-toc.action";
import {environment} from "@environments/environment";
import {
  Action,
  Actions,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {
  Observable,
  of,
} from "rxjs";
import {
  catchError,
  map,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  BaseStateModel,
  BasicState,
  defaultBaseStateInitValue,
  isNotNullNorUndefined,
  isNullOrUndefined,
  NotificationService,
  SolidifyError,
  StoreUtil,
  urlSeparator,
} from "solidify-frontend";

export interface AppTocStateModel extends BaseStateModel {
  toolsGuide: string | undefined;
  integrationGuide: string | undefined;
  userDocumentation: string | undefined;
  apis: string | undefined;
}

@Injectable()
@State<AppTocStateModel>({
  name: LocalStateEnum.application_toc,
  defaults: {
    ...defaultBaseStateInitValue(),
    toolsGuide: undefined,
    integrationGuide: undefined,
    userDocumentation: undefined,
    apis: undefined,
  },
})
export class AppTocState extends BasicState<AppTocStateModel> {
  constructor(protected httpClient: HttpClient,
              protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions) {
    super();
  }

  protected get _urlResource(): string {
    return BaseResourceApiEnum.docs;
  }

  @Action(AppTocAction.GetAllDocumentation)
  getAllDocumentation(ctx: StateContext<AppTocStateModel>, action: AppTocAction.GetAllDocumentation): Observable<any> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return StoreUtil.dispatchParallelActionAndWaitForSubActionsCompletion(
      ctx,
      [
        {action: new AppTocAction.GetIntegrationGuide()},
        {action: new AppTocAction.GetApis()},
        {action: new AppTocAction.GetToolsGuide()},
        {action: new AppTocAction.GetUserDocumentation()},
      ]).pipe(
      tap(success => {
        if (success) {
          ctx.dispatch(new AppTocAction.GetAllDocumentationSuccess(action));
        } else {
          ctx.dispatch(new AppTocAction.GetAllDocumentationFail(action));
        }
      }),
    );
  }

  @Action(AppTocAction.GetAllDocumentationSuccess)
  getAllDocumentationSuccess(ctx: StateContext<AppTocStateModel>, action: AppTocAction.GetAllDocumentationSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(AppTocAction.GetAllDocumentationFail)
  getAllDocumentationFail(ctx: StateContext<AppTocStateModel>, action: AppTocAction.GetAllDocumentationFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(AppTocAction.GetToolsGuide)
  getToolsGuide(ctx: StateContext<AppTocStateModel>, action: AppTocAction.GetToolsGuide): Observable<string> {
    if (isNullOrUndefined(environment.documentationTocToolsGuide)) {
      return of(undefined);
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    let headers = new HttpHeaders();
    headers = headers.set("Accept", ["application/xml"]);
    headers = headers.set("Content-Type", "text/xml");
    return this.httpClient.get(this._urlResource + urlSeparator + environment.documentationTocToolsGuide, {
      headers,
      observe: "response",
      responseType: "text",
    }).pipe(
      map((response: HttpResponse<string>) =>
        response.body),
      tap((toolsGuide: string) => {
        ctx.dispatch(new AppTocAction.GetToolsGuideSuccess(action, toolsGuide));
      }),
      catchError(error => {
        ctx.dispatch(new AppTocAction.GetToolsGuideFail(action));
        throw new SolidifyError("AppTocState GetToolsGuide", error);
      }),
    );
  }

  @Action(AppTocAction.GetToolsGuideSuccess)
  getToolsGuideSuccess(ctx: StateContext<AppTocStateModel>, action: AppTocAction.GetToolsGuideSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      toolsGuide: action.toolsGuide,
    });
  }

  @Action(AppTocAction.GetToolsGuideFail)
  getToolsGuideFail(ctx: StateContext<AppTocStateModel>, action: AppTocAction.GetToolsGuideFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(AppTocAction.GetIntegrationGuide)
  getIntegrationGuide(ctx: StateContext<AppTocStateModel>, action: AppTocAction.GetIntegrationGuide): Observable<string> {
    if (isNullOrUndefined(environment.documentationTocIntegrationGuide)) {
      return of(undefined);
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    let headers = new HttpHeaders();
    headers = headers.set("Accept", ["application/xml"]);
    headers = headers.set("Content-Type", "text/xml");

    return this.httpClient.get(this._urlResource + urlSeparator + environment.documentationTocIntegrationGuide, {
      headers,
      observe: "response",
      responseType: "text",
    }).pipe(
      map((response: HttpResponse<string>) =>
        response.body),
      tap((integrationGuide: string) => {
        ctx.dispatch(new AppTocAction.GetIntegrationGuideSuccess(action, integrationGuide));
      }),
      catchError(error => {
        ctx.dispatch(new AppTocAction.GetIntegrationGuideFail(action));
        throw new SolidifyError("AppTocState GetIntegrationGuide", error);
      }),
    );
  }

  @Action(AppTocAction.GetIntegrationGuideSuccess)
  getIntegrationGuideSuccess(ctx: StateContext<AppTocStateModel>, action: AppTocAction.GetIntegrationGuideSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      integrationGuide: action.integrationGuide,
    });
  }

  @Action(AppTocAction.GetIntegrationGuideFail)
  getIntegrationGuideFail(ctx: StateContext<AppTocStateModel>, action: AppTocAction.GetIntegrationGuideFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(AppTocAction.GetApis)
  getApis(ctx: StateContext<AppTocStateModel>, action: AppTocAction.GetApis): Observable<string> {
    if (isNullOrUndefined(environment.documentationTocApisGuide)) {
      return of(undefined);
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    let headers = new HttpHeaders();
    headers = headers.set("Accept", ["application/xml"]);
    headers = headers.set("Content-Type", "text/xml");

    return this.httpClient.get(this._urlResource + urlSeparator + environment.documentationTocApisGuide, {
      headers,
      observe: "response",
      responseType: "text",
    }).pipe(
      map((response: HttpResponse<string>) =>
        response.body),
      tap((apis: string) => {
        ctx.dispatch(new AppTocAction.GetApisSuccess(action, apis));
      }),
      catchError(error => {
        ctx.dispatch(new AppTocAction.GetApisFail(action));
        throw new SolidifyError("AppTocState GetApis", error);
      }),
    );
  }

  @Action(AppTocAction.GetApisSuccess)
  getApisSuccess(ctx: StateContext<AppTocStateModel>, action: AppTocAction.GetApisSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      apis: action.apis,
    });
  }

  @Action(AppTocAction.GetApisFail)
  getApisFail(ctx: StateContext<AppTocStateModel>, action: AppTocAction.GetApisFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(AppTocAction.GetUserDocumentation)
  getUserDocumentation(ctx: StateContext<AppTocStateModel>, action: AppTocAction.GetUserDocumentation): Observable<string> | Observable<undefined> {
    if (isNullOrUndefined(environment.documentationTocUserGuide)) {
      return of(undefined);
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    let headers = new HttpHeaders();
    headers = headers.set("Accept", ["application/xml"]);
    headers = headers.set("Content-Type", "text/xml");

    return this.httpClient.get((isNotNullNorUndefined(environment.documentationTocUserGuidePath) ? environment.documentationTocUserGuidePath : this._urlResource) + urlSeparator + environment.documentationTocUserGuide, {
      headers,
      observe: "response",
      responseType: "text",
    }).pipe(
      map((response: HttpResponse<string>) =>
        response.body),
      tap((userDocumentation: string) => {
        ctx.dispatch(new AppTocAction.GetUserDocumentationSuccess(action, userDocumentation));
      }),
      catchError(error => {
        ctx.dispatch(new AppTocAction.GetUserDocumentationFail(action));
        throw new SolidifyError("AppTocState GetUserDocumentation", error);
      }),
    );
  }

  @Action(AppTocAction.GetUserDocumentationSuccess)
  getUserDocumentationSuccess(ctx: StateContext<AppTocStateModel>, action: AppTocAction.GetUserDocumentationSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      userDocumentation: action.userDocumentation,
    });
  }

  @Action(AppTocAction.GetUserDocumentationFail)
  getUserDocumentationFail(ctx: StateContext<AppTocStateModel>, action: AppTocAction.GetUserDocumentationFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }
}
