import {Injectable} from "@angular/core";
import {AdminResourceApiEnum} from "@app/shared/enums/api.enum";
import {LocalStateEnum} from "@app/shared/enums/local-state.enum";
import {
  AppSystemPropertyAction,
  appSystemPropertyActionNameSpace,
} from "@app/stores/system-property/app-system-property.action";
import {environment} from "@environments/environment";
import {
  SystemProperty,
  User,
} from "@models";
import {
  Action,
  Actions,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  defaultResourceStateInitValue,
  isNullOrUndefined,
  NotificationService,
  QueryParameters,
  ResourceState,
  ResourceStateModel,
  SolidifyStateError,
} from "solidify-frontend";

export interface AppSystemPropertyStateModel extends ResourceStateModel<SystemProperty> {
}

@Injectable()
@State<AppSystemPropertyStateModel>({
  name: LocalStateEnum.application_systemProperty,
  defaults: {
    ...defaultResourceStateInitValue(),
    queryParameters: new QueryParameters(environment.defaultEnumValuePageSizeOption),
  },
})
export class AppSystemPropertyState extends ResourceState<AppSystemPropertyStateModel, SystemProperty> {
  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: appSystemPropertyActionNameSpace,
    });
  }

  protected get _urlResource(): string {
    return AdminResourceApiEnum.systemProperties;
  }

  @Action(AppSystemPropertyAction.GetSystemProperties)
  getSystemProperty(ctx: StateContext<AppSystemPropertyStateModel>, action: AppSystemPropertyAction.GetSystemProperties): Observable<SystemProperty> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    const url = AdminResourceApiEnum.systemProperties;
    if (isNullOrUndefined(environment.allowedUrls)) {
      environment.allowedUrls = [];
    }
    environment.allowedUrls.push(url);
    return this.apiService.getByIdInPath<SystemProperty>(url)
      .pipe(
        tap((systemProperty: SystemProperty) => {
          ctx.dispatch(new AppSystemPropertyAction.GetSystemPropertiesSuccess(action, systemProperty));
        }),
        catchError(error => {
          ctx.dispatch(new AppSystemPropertyAction.GetSystemPropertiesFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(AppSystemPropertyAction.GetSystemPropertiesSuccess)
  getSystemPropertiesSuccess(ctx: StateContext<AppSystemPropertyStateModel>, action: AppSystemPropertyAction.GetSystemPropertiesSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      current: action.systemProperty,
    });
  }

  @Action(AppSystemPropertyAction.GetSystemPropertiesFail)
  getSystemPropertiesFail(ctx: StateContext<AppSystemPropertyStateModel>, action: AppSystemPropertyAction.GetSystemPropertiesFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }
}
