import {Injectable} from "@angular/core";
import {AdminResourceApiEnum} from "@app/shared/enums/api.enum";
import {LocalStateEnum} from "@app/shared/enums/local-state.enum";
import {
  AppPersonAction,
  appPersonActionNameSpace,
} from "@app/stores/person/app-person.action";
import {Person} from "@models";
import {
  Action,
  Actions,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  defaultResourceStateInitValue,
  NotificationService,
  OverrideDefaultAction,
  QueryParameters,
  ResourceState,
  ResourceStateModel,
  SolidifyStateError,
} from "solidify-frontend";
import {environment} from "../../../environments/environment";

export interface AppPersonStateModel extends ResourceStateModel<Person> {
}

@Injectable()
@State<AppPersonStateModel>({
  name: LocalStateEnum.application_person,
  defaults: {
    ...defaultResourceStateInitValue(),
    queryParameters: new QueryParameters(environment.defaultEnumValuePageSizeOption),
  },
})
export class AppPersonState extends ResourceState<AppPersonStateModel, Person> {
  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: appPersonActionNameSpace,
      keepCurrentStateAfterUpdate: true,
    });
  }

  protected get _urlResource(): string {
    return AdminResourceApiEnum.people;
  }

  // Override method here to form to use url override
  @OverrideDefaultAction()
  @Action(AppPersonAction.GetById)
  getById(ctx: StateContext<AppPersonStateModel>, action: AppPersonAction.GetById): Observable<Person> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      current: null,
    });

    return this.apiService.getById<Person>(AdminResourceApiEnum.people, action.id)
      .pipe(
        tap((model: Person) => {
          ctx.dispatch(new AppPersonAction.GetByIdSuccess(action, model));
        }),
        catchError(error => {
          ctx.dispatch(new AppPersonAction.GetByIdFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(AppPersonAction.AddCurrentPerson)
  getCurrentUserSuccess(ctx: StateContext<AppPersonStateModel>, action: AppPersonAction.AddCurrentPerson): void {
    ctx.patchState({
      current: action.model,
    });
  }
}
