import {AdminOrganizationalUnitStateModel} from "@admin/organizational-unit/stores/admin-organizational-unit.state";
import {Injectable} from "@angular/core";
import {AppCartAction} from "@app/stores/cart/app-cart.action";
import {
  AppCartOrderAction,
  appCartOrderActionNameSpace,
} from "@app/stores/cart/order/app-cart-order.action";
import {Order} from "@models";
import {
  Action,
  Actions,
  Selector,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {ApiActionEnum} from "@shared/enums/api-action.enum";
import {AccessResourceApiEnum} from "@shared/enums/api.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {OrderArchive} from "@shared/models/business/order-archive.model";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  defaultResourceStateInitValue,
  isNullOrUndefined,
  NotificationService,
  OverrideDefaultAction,
  ResourceState,
  ResourceStateModel,
  SolidifyStateError,
  StoreUtil,
} from "solidify-frontend";

export interface AppCartOrderStateModel extends ResourceStateModel<Order> {
}

@Injectable()
@State<AppCartOrderStateModel>({
  name: LocalStateEnum.application_cart_order,
  defaults: {
    ...defaultResourceStateInitValue(),
  },
  children: [],
})
export class AppCartOrderState extends ResourceState<AppCartOrderStateModel, Order> {
  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: appCartOrderActionNameSpace,
    });
  }

  protected get _urlResource(): string {
    return AccessResourceApiEnum.orders;
  }

  @Selector()
  static isLoading(state: AppCartOrderStateModel): boolean {
    return StoreUtil.isLoadingState(state);
  }

  @Selector()
  static currentTitle(state: AppCartOrderStateModel): string | undefined {
    if (isNullOrUndefined(state.current)) {
      return undefined;
    }
    return state.current.name;
  }

  @Selector()
  static isLoadingWithDependency(state: AppCartOrderStateModel): boolean {
    return this.isLoading(state);
  }

  @Selector()
  static isReadyToBeDisplayed(state: AppCartOrderStateModel): boolean {
    return this.isReadyToBeDisplayedInCreateMode
      && !isNullOrUndefined(state.current);
  }

  @Selector()
  static isReadyToBeDisplayedInCreateMode(state: AppCartOrderStateModel): boolean {
    return true;
  }

  @OverrideDefaultAction()
  @Action(AppCartOrderAction.CreateSuccess)
  createSuccess(ctx: StateContext<AdminOrganizationalUnitStateModel>, action: AppCartOrderAction.CreateSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      current: action.model,
    });

    ctx.dispatch(new AppCartOrderAction.Save(action.model.resId));
  }

  @OverrideDefaultAction()
  @Action(AppCartOrderAction.CreateFail)
  createFail(ctx: StateContext<AdminOrganizationalUnitStateModel>, action: AppCartOrderAction.CreateFail): void {
    super.createFail(ctx, action);
    ctx.dispatch(new AppCartAction.SubmitFail(action as any, undefined));
  }

  @Action(AppCartOrderAction.Save)
  save(ctx: StateContext<AppCartOrderStateModel>, action: AppCartOrderAction.Save): Observable<OrderArchive> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return this.apiService.post<OrderArchive>(`${this._urlResource}/${action.resId}/${ApiActionEnum.SAVE}`)
      .pipe(
        tap(() => {
          ctx.dispatch(new AppCartOrderAction.SaveSuccess(action));
        }),
        catchError(error => {
          ctx.dispatch(new AppCartOrderAction.SaveFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(AppCartOrderAction.SaveSuccess)
  saveSuccess(ctx: StateContext<AppCartOrderStateModel>, action: AppCartOrderAction.SaveSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    ctx.dispatch(new AppCartOrderAction.Submit((action.parentAction as AppCartOrderAction.Save).resId));
  }

  @Action(AppCartOrderAction.SaveFail)
  saveFail(ctx: StateContext<AppCartOrderStateModel>, action: AppCartOrderAction.SaveFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });

    const resId = (action.parentAction as AppCartOrderAction.Save).resId;
    ctx.dispatch(new AppCartAction.SubmitFail(action as any, resId));
  }

  @Action(AppCartOrderAction.Submit)
  submit(ctx: StateContext<AppCartOrderStateModel>, action: AppCartOrderAction.Submit): Observable<string[]> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return this.apiService.post(`${this._urlResource}/${action.resId}/${ApiActionEnum.SUBMIT}`, null)
      .pipe(
        tap(() => {
          ctx.dispatch(new AppCartOrderAction.SubmitSuccess(action));
        }),
        catchError(error => {
          ctx.dispatch(new AppCartOrderAction.SubmitFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(AppCartOrderAction.SubmitSuccess)
  submitSuccess(ctx: StateContext<AppCartOrderStateModel>, action: AppCartOrderAction.SubmitSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    const resId = (action.parentAction as AppCartOrderAction.Submit).resId;
    ctx.dispatch(new AppCartAction.SubmitSuccess(action as any, resId));
  }

  @Action(AppCartOrderAction.SubmitFail)
  submitFail(ctx: StateContext<AppCartOrderStateModel>, action: AppCartOrderAction.SubmitFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    const resId = (action.parentAction as AppCartOrderAction.Submit).resId;
    ctx.dispatch(new AppCartAction.SubmitFail(action as any, resId));
  }

}
