import {HttpClient} from "@angular/common/http";
import {Injectable} from "@angular/core";
import {AccessResourceApiEnum} from "@app/shared/enums/api.enum";
import {LocalStateEnum} from "@app/shared/enums/local-state.enum";
import {
  AppCartDipAction,
  appCartDipActionNameSpace,
} from "@app/stores/cart/dip/app-cart-dip.action";
import {AppCartDipDataFileAction} from "@app/stores/cart/dip/data-file/app-cart-dip-data-file.action";
import {AppCartDipDataFileState} from "@app/stores/cart/dip/data-file/app-cart-dip-data-file.state";
import {Dip} from "@models";
import {
  Action,
  Actions,
  ofActionCompleted,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {ApiActionEnum} from "@shared/enums/api-action.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {DownloadService} from "@shared/services/download.service";
import {
  filter,
  take,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  defaultResourceStateInitValue,
  isFalse,
  NotificationService,
  ResourceState,
  ResourceStateModel,
} from "solidify-frontend";

export interface AppCartDipStateModel extends ResourceStateModel<Dip> {
}

@Injectable()
@State<AppCartDipStateModel>({
  name: LocalStateEnum.application_cart_dip,
  defaults: {
    ...defaultResourceStateInitValue(),
  },
  children: [
    AppCartDipDataFileState,
  ],
})
export class AppCartDipState extends ResourceState<AppCartDipStateModel, Dip> {
  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions,
              protected httpClient: HttpClient,
              private downloadService: DownloadService) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: appCartDipActionNameSpace,
    });
  }

  protected get _urlResource(): string {
    return AccessResourceApiEnum.dip;
  }

  @Action(AppCartDipAction.Download)
  download(ctx: StateContext<AppCartDipStateModel>, action: AppCartDipAction.Download): void {
    const fileName = "dip_" + action.id + ".zip";
    this.notificationService.showInformation(LabelTranslateEnum.fileDownloadStart);

    const actionDownload = new AppCartDipDataFileAction.GetAll(action.id);
    this.actions$.pipe(
      ofActionCompleted(AppCartDipDataFileAction.GetAllSuccess),
      filter(actionGetByIdSuccess => (actionGetByIdSuccess.action as AppCartDipDataFileAction.GetAllSuccess).parentAction === actionDownload),
      take(1),
      tap(actionGetAllSuccess => {
        if (isFalse(actionGetAllSuccess.result.successful)) {
          return;
        }
        const listDipDataFile = (actionGetAllSuccess.action as AppCartDipDataFileAction.GetAllSuccess).list;
        let size = 0;
        listDipDataFile._data.forEach(dipDataFile => {
          size += dipDataFile.fileSize;
        });
        this.downloadService.download(false, `${this._urlResource}/${action.id}/${ApiActionEnum.DL}`, fileName, size);
      }),
    ).subscribe();
    ctx.dispatch(actionDownload);
  }
}
