import {Injectable} from "@angular/core";
import {
  AppCartArchiveAction,
  appCartArchiveNamespace,
} from "@app/stores/cart/archive/app-cart-archive.action";
import {Archive} from "@home/models/archive.model";
import {Metadata} from "@models";
import {
  Action,
  Actions,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {AccessResourceApiEnum} from "@shared/enums/api.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {ArchiveHelper} from "@shared/helpers/archive.helper";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  defaultResourceStateInitValue,
  NotificationService,
  OverrideDefaultAction,
  ResourceState,
  ResourceStateModel,
  SolidifyStateError,
} from "solidify-frontend";

export interface AppCartArchiveStateModel extends ResourceStateModel<Archive> {
}

@Injectable()
@State<AppCartArchiveStateModel>({
  name: LocalStateEnum.application_cart_archive,
  defaults: {
    ...defaultResourceStateInitValue(),
  },
})
export class AppCartArchiveState extends ResourceState<AppCartArchiveStateModel, Archive> {
  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: appCartArchiveNamespace,
    });
  }

  protected get _urlResource(): string {
    return AccessResourceApiEnum.publicMetadata;
  }

  @OverrideDefaultAction()
  @Action(AppCartArchiveAction.GetById)
  getById(ctx: StateContext<AppCartArchiveStateModel>, action: AppCartArchiveAction.GetById): Observable<Archive> {
    let reset = {};
    if (!action.keepCurrentContext) {
      reset = {
        current: undefined,
      };
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      ...reset,
    });

    return this.apiService.getById<Metadata>(this._urlResource, action.id)
      .pipe(
        tap((model: any) => {
          ctx.dispatch(new AppCartArchiveAction.GetByIdSuccess(action, ArchiveHelper.adaptArchiveMetadataInArchive(model)));
        }),
        catchError(error => {
          ctx.dispatch(new AppCartArchiveAction.GetByIdFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @OverrideDefaultAction()
  @Action(AppCartArchiveAction.GetByListId)
  getByListId(ctx: StateContext<AppCartArchiveStateModel>, action: AppCartArchiveAction.GetByListId): void {
    ctx.patchState({
      list: undefined,
    });
    super.getByListId(ctx, action);
  }

  @OverrideDefaultAction()
  @Action(AppCartArchiveAction.GetByListIdSuccess)
  getByListIdSuccess(ctx: StateContext<AppCartArchiveStateModel>, action: AppCartArchiveAction.GetByListIdSuccess): void {
    super.getByListIdSuccess(ctx, action);
    const list = ctx.getState().list;
    ctx.patchState({
      total: list ? list.length : 0,
    });
  }

  @Action(AppCartArchiveAction.RemoveAllInList)
  deleteAllInList(ctx: StateContext<AppCartArchiveStateModel>, action: AppCartArchiveAction.RemoveAllInList): void {
    ctx.patchState({
      list: [],
      total: 0,
    });
  }
}
