import {Injectable} from "@angular/core";
import {AppCartAction} from "@app/stores/cart/app-cart.action";
import {AppCartArchiveAction} from "@app/stores/cart/archive/app-cart-archive.action";
import {
  AppCartArchiveState,
  AppCartArchiveStateModel,
} from "@app/stores/cart/archive/app-cart-archive.state";
import {
  AppCartDipState,
  AppCartDipStateModel,
} from "@app/stores/cart/dip/app-cart-dip.state";
import {AppCartOrderAction} from "@app/stores/cart/order/app-cart-order.action";
import {
  AppCartOrderState,
  AppCartOrderStateModel,
} from "@app/stores/cart/order/app-cart-order.state";
import {Enums} from "@enums";
import {HomeStateModel} from "@home/stores/home.state";
import {Order} from "@models";
import {Navigate} from "@ngxs/router-plugin";
import {
  Action,
  Actions,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {RoutesEnum} from "@shared/enums/routes.enum";
import {SessionStorageEnum} from "@shared/enums/session-storage.enum";
import {SessionStorageHelper} from "@shared/helpers/session-storage.helper";
import {
  ApiService,
  BaseStateModel,
  isNullOrUndefined,
  MARK_AS_TRANSLATABLE,
  NotificationService,
  urlSeparator,
} from "solidify-frontend";

export interface AppCartStateModel extends BaseStateModel {
  application_cart_archive: AppCartArchiveStateModel | undefined;
  application_cart_order: AppCartOrderStateModel | undefined;
  application_cart_dip: AppCartDipStateModel | undefined;
}

@Injectable()
@State<AppCartStateModel>({
  name: LocalStateEnum.application_cart,
  defaults: {
    application_cart_archive: undefined,
    application_cart_order: undefined,
    application_cart_dip: undefined,
    isLoadingCounter: 0,
  },
  children: [
    AppCartArchiveState,
    AppCartOrderState,
    AppCartDipState,
  ],
})
export class AppCartState {
  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions) {
  }

  @Action(AppCartAction.AddToCart)
  addToCart(ctx: StateContext<HomeStateModel>, action: AppCartAction.AddToCart): void {
    const numberBefore = SessionStorageHelper.getListItem(SessionStorageEnum.cart).length;
    const numberAfter = SessionStorageHelper.addItemInList(SessionStorageEnum.cart, action.archive.resId).length;
    if (numberBefore < numberAfter) {
      ctx.dispatch(new AppCartArchiveAction.AddInList(action.archive));
      this.notificationService.showSuccess(MARK_AS_TRANSLATABLE("home.archive.cart.notification.addWithSuccessInCart"));
    } else {
      this.notificationService.showInformation(MARK_AS_TRANSLATABLE("home.archive.cart.notification.alreadyInCart"));
    }
  }

  @Action(AppCartAction.RemoveToCart)
  removeToCart(ctx: StateContext<HomeStateModel>, action: AppCartAction.RemoveToCart): void {
    SessionStorageHelper.removeItemInList(SessionStorageEnum.cart, action.resId);
    ctx.dispatch(new AppCartArchiveAction.RemoveInListById(action.resId));
  }

  @Action(AppCartAction.RemoveAll)
  removeAll(ctx: StateContext<HomeStateModel>, action: AppCartAction.RemoveAll): void {
    SessionStorageHelper.initItemInList(SessionStorageEnum.cart, undefined);
    ctx.dispatch(new AppCartArchiveAction.RemoveAllInList());
  }

  @Action(AppCartAction.GetAllCart)
  getAllCart(ctx: StateContext<HomeStateModel>, action: AppCartAction.GetAllCart): void {
    const listArchiveId = SessionStorageHelper.getListItem(SessionStorageEnum.cart);
    ctx.dispatch(new AppCartArchiveAction.GetByListId(listArchiveId));
  }

  @Action(AppCartAction.Submit)
  submit(ctx: StateContext<HomeStateModel>, action: AppCartAction.Submit): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    const order = {
      name: action.order.name,
      queryType: Enums.Order.QueryTypeEnum.SIMPLE,
      query: sessionStorage.getItem(SessionStorageEnum.cart),
    } as Order;
    ctx.dispatch(new AppCartOrderAction.Create({
      model: order,
    }));
  }

  @Action(AppCartAction.SubmitSuccess)
  submitSuccess(ctx: StateContext<HomeStateModel>, action: AppCartAction.SubmitSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });

    this.notificationService.showSuccess(MARK_AS_TRANSLATABLE("app.order.notification.cartSubmit.success"));

    SessionStorageHelper.addItemInList(SessionStorageEnum.orderPending, action.orderId);

    ctx.dispatch([
      new Navigate([RoutesEnum.orderMyOrderDetail + urlSeparator + action.orderId]),
      new AppCartAction.Clean(),
    ]);
  }

  @Action(AppCartAction.SubmitFail)
  submitFail(ctx: StateContext<HomeStateModel>, action: AppCartAction.SubmitFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });

    this.notificationService.showError(MARK_AS_TRANSLATABLE("app.order.notification.cartSubmit.fail"));
    if (isNullOrUndefined(action.orderId)) {
      return;
    }
    ctx.dispatch(new Navigate([RoutesEnum.orderMyOrderDetail + urlSeparator + action.orderId]));
  }

  @Action(AppCartAction.Clean)
  clean(ctx: StateContext<HomeStateModel>, action: AppCartAction.Clean): void {
    sessionStorage.removeItem(SessionStorageEnum.cart);
    ctx.dispatch(new AppCartArchiveAction.Clean(false));
  }
}
