import {
  HttpClient,
  HttpHeaders,
  HttpResponse,
} from "@angular/common/http";
import {Injectable} from "@angular/core";
import {LocalStateEnum} from "@app/shared/enums/local-state.enum";
import {AppState} from "@app/stores/app.state";
import {AppCarouselAction} from "@app/stores/carousel/app-carousel.action";
import {environment} from "@environments/environment";
import {
  Action,
  Actions,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {Observable} from "rxjs";
import {
  catchError,
  map,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  BaseStateModel,
  BasicState,
  defaultBaseStateInitValue,
  isNonEmptyString,
  isNullOrUndefined,
  MemoizedUtil,
  NotificationService,
  SolidifyError,
} from "solidify-frontend";

export interface AppCarouselStateModel extends BaseStateModel {
  tiles: Node[] | undefined;
}

@Injectable()
@State<AppCarouselStateModel>({
  name: LocalStateEnum.application_carousel,
  defaults: {
    ...defaultBaseStateInitValue(),
    tiles: undefined,
  },
})
export class AppCarouselState extends BasicState<AppCarouselStateModel> {
  constructor(protected httpClient: HttpClient,
              protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions) {
    super();
  }

  @Action(AppCarouselAction.GetCarousel)
  getCarousel(ctx: StateContext<AppCarouselStateModel>, action: AppCarouselAction.GetCarousel): Observable<string> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    let language = MemoizedUtil.selectSnapshot(this.store, AppState, state => state.appLanguage);
    if (isNullOrUndefined(language)) {
      language = environment.defaultLanguage;
    }
    let headers = new HttpHeaders();
    headers = headers.set("Accept", ["application/xml"]);
    headers = headers.set("Content-Type", "text/xml");

    let url = `assets/themes/${environment.theme}/carousel/`;
    if (isNonEmptyString(environment.carouselUrl)) {
      url = environment.carouselUrl;
    }
    return this.httpClient.get(url + `carousel_${language}.html`, {
      headers,
      observe: "response",
      responseType: "text",
    }).pipe(
      map((response: HttpResponse<string>) => response.body),
      tap((htmlContent: string) => {
        const parser = new DOMParser();
        const parsedHtml = parser.parseFromString(htmlContent, "text/html");
        const nodeList = parsedHtml.querySelectorAll("body > ul > li");
        const listTiles = [];
        nodeList.forEach(node => {
          listTiles.push(node);
        });
        ctx.dispatch(new AppCarouselAction.GetCarouselSuccess(action, listTiles));
      }),
      catchError(error => {
        ctx.dispatch(new AppCarouselAction.GetCarouselFail(action));
        throw new SolidifyError("AppCarouselAction GetCarousel", error);
      }),
    );
  }

  @Action(AppCarouselAction.GetCarouselSuccess)
  getCarouselSuccess(ctx: StateContext<AppCarouselStateModel>, action: AppCarouselAction.GetCarouselSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      tiles: action.tiles,
    });
  }

  @Action(AppCarouselAction.GetCarouselFail)
  getCarouselFail(ctx: StateContext<AppCarouselStateModel>, action: AppCarouselAction.GetCarouselFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }
}
