import {PersonRole} from "@admin/models/person-role.model";
import {HttpClient} from "@angular/common/http";
import {Injectable} from "@angular/core";
import {RoutesEnum} from "@app/shared/enums/routes.enum";
import {SessionStorageEnum} from "@app/shared/enums/session-storage.enum";
import {ThemeEnum} from "@app/shared/enums/theme.enum";
import {JwtTokenHelper} from "@app/shared/helpers/jwt-token.helper";
import {UserPreferencesUtil} from "@app/shared/utils/user-preferences.util";
import {AppAction} from "@app/stores/app.action";
import {AppArchiveAclAction} from "@app/stores/archive-acl/app-archive-acl.action";
import {
  AppArchiveAclState,
  AppArchiveAclStateModel,
} from "@app/stores/archive-acl/app-archive-acl.state";
import {AppAuthorizedOrganizationalUnitAction} from "@app/stores/authorized-organizational-unit/app-authorized-organizational-unit.action";
import {
  AppAuthorizedOrganizationalUnitState,
  AppAuthorizedOrganizationalUnitStateModel,
} from "@app/stores/authorized-organizational-unit/app-authorized-organizational-unit.state";
import {
  AppBannerState,
  AppBannerStateModel,
} from "@app/stores/banner/app-banner.state";
import {AppCarouselAction} from "@app/stores/carousel/app-carousel.action";
import {
  AppCarouselState,
  AppCarouselStateModel,
} from "@app/stores/carousel/app-carousel.state";
import {AppCartAction} from "@app/stores/cart/app-cart.action";
import {AppCartState} from "@app/stores/cart/app-cart.state";
import {AppCartOrderAction} from "@app/stores/cart/order/app-cart-order.action";
import {AppCartOrderState} from "@app/stores/cart/order/app-cart-order.state";
import {AppNotificationInboxAction} from "@app/stores/notification-inbox/app-notification-inbox.action";
import {AppNotificationInboxState} from "@app/stores/notification-inbox/app-notification-inbox.state";
import {
  AppOrganizationalUnitPersonRoleState,
  AppOrganizationalUnitPersonRoleStateModel,
} from "@app/stores/organizational-unit-person-role/app-organizational-unit-person-role.state";
import {
  AppPersonState,
  AppPersonStateModel,
} from "@app/stores/person/app-person.state";
import {AppSystemPropertyAction} from "@app/stores/system-property/app-system-property.action";
import {
  AppSystemPropertyState,
  AppSystemPropertyStateModel,
} from "@app/stores/system-property/app-system-property.state";
import {AppTocAction} from "@app/stores/toc/app-toc.action";
import {
  AppTocState,
  AppTocStateModel,
} from "@app/stores/toc/app-toc.state";
import {AppUserAction} from "@app/stores/user/app-user.action";
import {
  AppUserState,
  AppUserStateModel,
} from "@app/stores/user/app-user.state";
import {Enums} from "@enums";
import {environment} from "@environments/environment";
import {
  ApplicationRole,
  NotificationDlcm,
  OrganizationalUnit,
  Person,
  Role,
  User,
} from "@models";
import {TranslateService} from "@ngx-translate/core";
import {Navigate} from "@ngxs/router-plugin";
import {
  Action,
  Actions,
  ofActionCompleted,
  ofActionErrored,
  Selector,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {ApiActionEnum} from "@shared/enums/api-action.enum";
import {ApiResourceNameEnum} from "@shared/enums/api-resource-name.enum";
import {
  AdminResourceApiEnum,
  BaseResourceApiEnum,
} from "@shared/enums/api.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {PollingHelper} from "@shared/helpers/polling.helper";
import {SessionStorageHelper} from "@shared/helpers/session-storage.helper";
import {ShortDoi} from "@shared/models/short-doi.model";
import {Token} from "@shared/models/token.model";
import * as moment from "moment";
import {Observable} from "rxjs";
import {
  catchError,
  mergeMap,
  take,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  BaseStateModel,
  BasicState,
  ClipboardUtil,
  defaultBaseStateInitValue,
  isFalse,
  isNullOrUndefined,
  isTrue,
  MappingObjectUtil,
  MARK_AS_TRANSLATABLE,
  MemoizedUtil,
  NotificationService,
  OAuth2Service,
  QueryParameters,
  StoreUtil,
  urlSeparator,
} from "solidify-frontend";

export interface AppStateModel extends BaseStateModel {
  isTouchInterface: boolean;
  isApplicationInitialized: boolean;
  isServerOffline: boolean;
  isLoggedIn: boolean;
  isInTourMode: boolean;
  ignorePreventLeavePopup: boolean;
  isOpenedSidebarUserGuide: boolean;
  userRoles: Enums.UserApplicationRole.UserApplicationRoleEnum[];
  appLanguage: Enums.Language.LanguageEnum;
  theme: ThemeEnum;
  preventExit: boolean;
  preventExitReasonToTranslate: string | undefined;
  application_user: AppUserStateModel | undefined;
  application_person: AppPersonStateModel | undefined;
  application_authorizedOrganizationalUnit: AppAuthorizedOrganizationalUnitStateModel | undefined;
  application_organizationalUnit_personRole: AppOrganizationalUnitPersonRoleStateModel | undefined;
  application_toc: AppTocStateModel | undefined;
  token: Token | undefined;
  application_systemProperty: AppSystemPropertyStateModel;
  displayCookieConsent: boolean;
  [LocalStateEnum.application_banner]: AppBannerStateModel | undefined;
  [LocalStateEnum.application_carousel]: AppCarouselStateModel | undefined;
  [LocalStateEnum.application_archiveAcl]: AppArchiveAclStateModel | undefined;
}

@Injectable()
@State<AppStateModel>({
  name: LocalStateEnum.application,
  defaults: {
    ...defaultBaseStateInitValue(),
    isTouchInterface: false,
    isApplicationInitialized: false,
    isServerOffline: false,
    isLoggedIn: false,
    isInTourMode: false,
    ignorePreventLeavePopup: false,
    isOpenedSidebarUserGuide: false,
    userRoles: [Enums.UserApplicationRole.UserApplicationRoleEnum.guest],
    appLanguage: undefined,
    theme: environment.theme,
    preventExit: false,
    preventExitReasonToTranslate: undefined,
    application_user: undefined,
    application_person: undefined,
    application_authorizedOrganizationalUnit: undefined,
    application_organizationalUnit_personRole: undefined,
    application_toc: undefined,
    token: undefined,
    application_systemProperty: undefined,
    displayCookieConsent: false,
    [LocalStateEnum.application_banner]: undefined,
    [LocalStateEnum.application_carousel]: undefined,
    [LocalStateEnum.application_archiveAcl]: undefined,
  },
  children: [
    AppUserState,
    AppPersonState,
    AppAuthorizedOrganizationalUnitState,
    AppOrganizationalUnitPersonRoleState,
    AppCartState,
    AppTocState,
    AppNotificationInboxState,
    AppSystemPropertyState,
    AppBannerState,
    AppCarouselState,
    AppArchiveAclState,
  ],
})
export class AppState extends BasicState<AppStateModel> {
  constructor(private store: Store,
              private translate: TranslateService,
              private oauthService: OAuth2Service,
              private actions$: Actions,
              private apiService: ApiService,
              private httpClient: HttpClient,
              private notificationService: NotificationService,
              private readonly _actions$: Actions) {
    super();
  }

  @Selector()
  static listAuthorizedOrganizationalUnit(state: AppStateModel): OrganizationalUnit[] {
    return state.application_authorizedOrganizationalUnit.list;
  }

  @Selector()
  static listAuthorizedOrganizationalUnitId(state: AppStateModel): string[] {
    const list = this.listAuthorizedOrganizationalUnit(state);
    if (isNullOrUndefined(list)) {
      return [];
    }
    return list.map(o => o.resId);
  }

  @Selector()
  static currentPerson(state: AppStateModel): Person {
    return state.application_person.current;
  }

  @Selector()
  static currentUser(state: AppStateModel): User {
    return state.application_user.current;
  }

  @Selector()
  static currentUserApplicationRole(state: AppStateModel): ApplicationRole {
    const currentPersonRole = this.currentUser(state);
    if (isNullOrUndefined(currentPersonRole)) {
      return undefined;
    }
    return currentPersonRole.applicationRole;
  }

  @Selector()
  static currentUserApplicationRoleResId(state: AppStateModel): string | undefined {
    const applicationRole = this.currentUserApplicationRole(state);
    if (isNullOrUndefined(applicationRole)) {
      return undefined;
    }
    return applicationRole.resId;
  }

  @Selector()
  static currentOrgUnitPerson(state: AppStateModel): PersonRole | undefined {
    if (isNullOrUndefined(state.application_organizationalUnit_personRole.current)) {
      return undefined;
    }
    return state.application_organizationalUnit_personRole.current;
  }

  @Selector()
  static currentOrgUnitPersonRole(state: AppStateModel): Role[] | undefined {
    const currentPerson = this.currentOrgUnitPerson(state);
    if (isNullOrUndefined(currentPerson)) {
      return undefined;
    }
    return currentPerson.roles;
  }

  @Selector()
  static currentOrgUnitPersonRoleResId(state: AppStateModel): string[] | undefined {
    const currentPersonRole = this.currentOrgUnitPersonRole(state);
    if (isNullOrUndefined(currentPersonRole)) {
      return undefined;
    }
    return currentPersonRole.map(r => r.resId);
  }

  @Action(AppAction.InitApplication)
  initApplication(ctx: StateContext<AppStateModel>, action: AppAction.InitApplication): Observable<any> {
    return StoreUtil.dispatchSequentialActionAndWaitForSubActionsCompletion(
      ctx,
      [
        {
          action: new AppAction.LoadModuleUrl(),
          subActionCompletions: [
            this.actions$.pipe(ofActionCompleted(AppAction.LoadModuleUrlSuccess)),
            this.actions$.pipe(ofActionCompleted(AppAction.LoadModuleUrlFail)),
          ],
        },
        {
          action: new AppAction.Login(),
          subActionCompletions: [
            this.actions$.pipe(ofActionCompleted(AppAction.LoginSuccess)),
            this.actions$.pipe(ofActionErrored(AppAction.LoginSuccess)),
            this.actions$.pipe(ofActionCompleted(AppAction.LoginFail)),
          ],
        },
        {
          action: new AppAction.InitApplicationParallel(),
          subActionCompletions: [
            this.actions$.pipe(ofActionCompleted(AppAction.InitApplicationParallelSuccess)),
          ],
        },
      ],
    ).pipe(
      tap(isAuthenticatedUser => {
        ctx.dispatch(new AppAction.InitApplicationSuccess(action));
      }),
    );
  }

  @Action(AppAction.InitApplicationParallel)
  initApplicationParallel(ctx: StateContext<AppStateModel>, action: AppAction.InitApplicationParallel): Observable<any> {
    return StoreUtil.dispatchParallelActionAndWaitForSubActionsCompletion(
      ctx,
      [
        {
          action: new AppAction.ChangeAppLanguage(UserPreferencesUtil.getPreferredLanguage()),
          subActionCompletions: [
            this.actions$.pipe(ofActionCompleted(AppAction.ChangeAppLanguageSuccess)),
          ],
        },
        {action: new AppAction.DefineFallbackLanguage()},
        {action: new AppAction.SetMomentLocal()},
        {action: new AppAction.LoadCart()}, // TODO See if should not be differed
        {action: new AppAction.StartPollingOrder()},
        {action: new AppAction.StartPollingNotification()},
        {action: new AppTocAction.GetAllDocumentation()},
        {
          action: new AppSystemPropertyAction.GetSystemProperties(),
          subActionCompletions: [
            this.actions$.pipe(ofActionCompleted(AppSystemPropertyAction.GetSystemPropertiesSuccess)),
            this.actions$.pipe(ofActionCompleted(AppSystemPropertyAction.GetSystemPropertiesFail)),
          ],
        },
      ],
    ).pipe(
      tap(success => {
        ctx.dispatch(new AppAction.InitApplicationParallelSuccess(action));
      }),
    );
  }

  @Action(AppAction.LoadModuleUrl)
  loadModuleUrl(ctx: StateContext<AppStateModel>, action: AppAction.LoadModuleUrl): Observable<any> {
    const url = BaseResourceApiEnum.preservationPlanning + urlSeparator + ApiResourceNameEnum.MODULE;
    if (isNullOrUndefined(environment.allowedUrls)) {
      environment.allowedUrls = [];
    }
    environment.allowedUrls.push(url);
    return this.apiService.getByIdInPath<string[]>(url)
      .pipe(
        tap(runtimeConfig => {
          // throw new Error("SIMULATE SERVER OFFLINE");
          ctx.dispatch(new AppAction.LoadModuleUrlSuccess(action, runtimeConfig));
        }),
        catchError(error => {
          ctx.dispatch(new AppAction.LoadModuleUrlFail(action));
          throw error;
        }),
      );
  }

  @Action(AppAction.LoadModuleUrlSuccess)
  loadModuleUrlSuccess(ctx: StateContext<AppStateModel>, action: AppAction.LoadModuleUrlSuccess): void {
    Object.assign(environment, action.runtimeConfig);
    if (!environment.computeOAuthEndpoint) {
      return;
    }
    environment.tokenEndpoint = environment.authorization + urlSeparator + "token";
    environment.loginUrl = environment.authorization + urlSeparator + "authorize";
    environment.allowedUrls.push(environment.tokenEndpoint, environment.loginUrl);
  }

  @Action(AppAction.LoadModuleUrlFail)
  loadModuleUrlFail(ctx: StateContext<AppStateModel>, action: AppAction.LoadModuleUrlFail): void {
    ctx.patchState({
      isServerOffline: true,
    });
    ctx.dispatch(new Navigate([RoutesEnum.serverOffline]));
  }

  @Action(AppAction.InitApplicationSuccess)
  initApplicationSuccess(ctx: StateContext<AppStateModel>): void {
    ctx.patchState({
      isApplicationInitialized: true,
    });
  }

  @Action(AppAction.DefineFallbackLanguage)
  defineFallbackLanguage(ctx: StateContext<AppStateModel>): void {
    // this language will be used as a fallback when a translation isn't found in the current language
    this.translate.setDefaultLang(environment.defaultLanguage);
  }

  @Action(AppAction.SetMomentLocal)
  setMomentLocal(ctx: StateContext<AppStateModel>): void {
    const navigatorLanguage = navigator.language;
    moment.locale(navigatorLanguage);
    // this._adapter.setLocale("fr-CH"); // TOFIX : DOESN'T WORK HERE, NEED TO BE SET IN APP MODULE
  }

  @Action(AppAction.ChangeAppLanguage)
  changeAppLanguage(ctx: StateContext<AppStateModel>, action: AppAction.ChangeAppLanguage): Observable<any> {
    ctx.patchState({
      appLanguage: action.language,
    });

    sessionStorage.setItem(SessionStorageEnum.language, action.language);
    return this.translate.use(action.language).pipe(
      tap(() => {
        ctx.dispatch(new AppAction.ChangeAppLanguageSuccess(action));
      }),
    );
  }

  @Action(AppAction.ChangeAppLanguageSuccess)
  changeAppLanguageSuccess(ctx: StateContext<AppStateModel>, action: AppAction.ChangeAppLanguageSuccess): void {
    ctx.dispatch(new AppCarouselAction.GetCarousel());
  }

  @Action(AppAction.ChangeAppTheme)
  changeAppTheme(ctx: StateContext<AppStateModel>, action: AppAction.ChangeAppTheme): void {
    environment.theme = action.theme;
    ctx.patchState({
      theme: action.theme,
    });
    // Force to refresh language file consolidated with theme json and customizable json
    this.translate.reloadLang(Enums.Language.LanguageEnum.en);
    this.translate.reloadLang(Enums.Language.LanguageEnum.fr);
    this.translate.reloadLang(Enums.Language.LanguageEnum.de);
    ctx.dispatch(new AppCarouselAction.GetCarousel());
  }

  @Action(AppAction.ChangeDisplaySidebarUserGuide)
  changeDisplaySidebarUserGuide(ctx: StateContext<AppStateModel>, action: AppAction.ChangeDisplaySidebarUserGuide): void {
    ctx.patchState({
      isOpenedSidebarUserGuide: action.isOpen,
    });
  }

  @Action(AppAction.LoadCart)
  loadCart(ctx: StateContext<AppStateModel>, action: AppAction.LoadCart): void {
    ctx.dispatch(new AppCartAction.GetAllCart());
  }

  @Action(AppAction.StartPollingOrder)
  startPollingOrder(ctx: StateContext<AppStateModel>, action: AppAction.StartPollingOrder): void {
    PollingHelper.startPollingObs({
      initialIntervalRefreshInSecond: environment.refreshOrderAvailableIntervalInSecond,
      incrementInterval: true,
      resetIntervalWhenUserMouseEvent: true,
      maximumIntervalRefreshInSecond: environment.refreshOrderAvailableIntervalInSecond * 10,
      filter: () => SessionStorageHelper.getListItem(SessionStorageEnum.orderPending).length > 0,
      actionToDo: () => {
        const listOrderPending = SessionStorageHelper.getListItem(SessionStorageEnum.orderPending);
        const orderList = MemoizedUtil.listSnapshot(this.store, AppCartOrderState);
        listOrderPending.forEach(orderId => {
          let order = undefined;
          if (!isNullOrUndefined(orderList)) {
            order = orderList.find(o => o.resId === orderId);
          }
          if (!isNullOrUndefined(order) && (order.status === Enums.Order.StatusEnum.READY || order.status === Enums.Order.StatusEnum.INERROR)) {
            SessionStorageHelper.removeItemInList(SessionStorageEnum.orderPending, orderId);
            SessionStorageHelper.addItemInList(SessionStorageEnum.newOrderAvailable, orderId);
            const actionNotification = {
              text: LabelTranslateEnum.see,
              callback: () => this.store.dispatch(new Navigate([RoutesEnum.orderMyOrderDetail, orderId])),
            };
            if (order.status === Enums.Order.StatusEnum.READY) {
              this.notificationService.showSuccess(MARK_AS_TRANSLATABLE("app.notification.order.ready"), undefined, actionNotification);
            } else {
              this.notificationService.showError(MARK_AS_TRANSLATABLE("app.notification.order.inError"), undefined, actionNotification);
            }
          } else {
            ctx.dispatch(new AppCartOrderAction.AddInListById(orderId, true, true));
          }
        });
      },
    }).subscribe();
  }

  @Action(AppAction.StartPollingNotification)
  startPollingNotification(ctx: StateContext<AppStateModel>, action: AppAction.StartPollingNotification): void {
    if (ctx.getState().isLoggedIn) {
      this.store.dispatch(new AppAction.UpdateNotificationInbox);
    }

    PollingHelper.startPollingObs({
      initialIntervalRefreshInSecond: environment.refreshNotificationInboxAvailableIntervalInSecond,
      incrementInterval: true,
      resetIntervalWhenUserMouseEvent: true,
      maximumIntervalRefreshInSecond: environment.refreshNotificationInboxAvailableIntervalInSecond * 10,
      filter: () => ctx.getState().isLoggedIn,
      actionToDo: () => {
        this.store.dispatch(new AppAction.UpdateNotificationInbox);
      },
    }).subscribe();
  }

  @Action(AppAction.UpdateNotificationInbox)
  updateNotificationInbox(ctx: StateContext<AppStateModel>, action: AppAction.UpdateNotificationInbox): void {
    const queryParameters = new QueryParameters(environment.maximalPageSizeToRetrievePaginationInfo);
    const searchItems = queryParameters.search.searchItems;
    const NOTIFICATION_STATUS: keyof NotificationDlcm = "notificationStatus";
    MappingObjectUtil.set(searchItems, NOTIFICATION_STATUS, Enums.Notification.StatusEnum.PENDING);

    this._actions$.pipe(
      ofActionCompleted(AppNotificationInboxAction.GetAllSuccess),
      take(1),
      tap(result => {
        if (!result.result.successful) {
          return;
        }
        const actionSuccess = (result.action as AppNotificationInboxAction.GetAllSuccess);
        const listNewNotificationInboxPending = actionSuccess.list._data.map(newNotification => newNotification.resId);
        const listOldNotificationInboxPending = SessionStorageHelper.getListItem(SessionStorageEnum.notificationInboxPending);
        let counterNewNotification = 0;
        listNewNotificationInboxPending.forEach(newNotificationId => {
          if (!listOldNotificationInboxPending.includes(newNotificationId)) {
            counterNewNotification++;
          }
        });
        if (counterNewNotification > 0) {
          const actionNotification = {
            text: LabelTranslateEnum.see,
            callback: () => this.store.dispatch(new Navigate([RoutesEnum.preservationSpaceNotificationInbox])),
          };
          this.notificationService.showSuccess(MARK_AS_TRANSLATABLE("app.notification.notificationInbox.newIncoming"), {"number": counterNewNotification}, actionNotification);
        }
        SessionStorageHelper.initItemInList(SessionStorageEnum.notificationInboxPending, listNewNotificationInboxPending);
      }),
    ).subscribe();

    this.store.dispatch(new AppNotificationInboxAction.GetAll(queryParameters));
  }

  @Action(AppAction.Login)
  login(ctx: StateContext<AppStateModel>, action: AppAction.Login): Observable<boolean> {
    return this.oauthService.tryLogin()
      .pipe(
        tap(e => {
            if (e) {
              ctx.dispatch(new AppAction.LoginSuccess(action));
            } else {
              ctx.dispatch(new AppAction.LoginFail(action));
            }
          },
        ));
  }

  @Action(AppAction.LoginSuccess)
  loginSuccess(ctx: StateContext<AppStateModel>): Observable<any> {
    const accessToken = this.oauthService.getAccessToken();
    const tokenDecoded = JwtTokenHelper.decodeToken(accessToken);
    ctx.patchState({
      isLoggedIn: true,
      userRoles: tokenDecoded.authorities,
      token: tokenDecoded,
    });
    return StoreUtil.dispatchParallelActionAndWaitForSubActionsCompletion(ctx, [
      {
        action: new AppUserAction.GetCurrentUser(),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(AppUserAction.GetCurrentUserSuccess)),
          this.actions$.pipe(ofActionCompleted(AppUserAction.GetCurrentUserFail)),
        ],
      },
      {
        action: new AppAuthorizedOrganizationalUnitAction.GetAll(),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(AppAuthorizedOrganizationalUnitAction.GetAllSuccess)),
          this.actions$.pipe(ofActionCompleted(AppAuthorizedOrganizationalUnitAction.GetAllFail)),
        ],
      },
      {
        action: new AppArchiveAclAction.GetAll(),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(AppArchiveAclAction.GetAllSuccess)),
          this.actions$.pipe(ofActionCompleted(AppArchiveAclAction.GetAllFail)),
        ],
      },
    ]).pipe(
      tap(success => {
      }),
    );
  }

  @Action(AppAction.LoginFail)
  loginFail(ctx: StateContext<AppStateModel>): void {
    ctx.patchState({
      isLoggedIn: false,
    });
  }

  @Action(AppAction.Logout)
  logout(ctx: StateContext<AppStateModel>): Observable<any> {
    const isLoggedIn = ctx.getState().isLoggedIn;
    if (isFalse(isLoggedIn)) {
      this.oauthService.logOut(true);
      return; // PREVENT INFINITE CANCEL LOOP DUE TO MULTIPLE CALL TO THIS METHOD THAT REDIRECT TO HOME PAGE
    }
    ctx.patchState({
      ignorePreventLeavePopup: true,
      isLoggedIn: false,
      userRoles: [Enums.UserApplicationRole.UserApplicationRoleEnum.guest],
      token: undefined,
      application_user: undefined,
    });

    // Revoke tokens servers side
    return this.apiService.post(AdminResourceApiEnum.users + urlSeparator + ApiActionEnum.REVOKE_MY_TOKENS).pipe(
      tap(() => {
        // Logout client side
        this.oauthService.logOut(true);
      }),
      mergeMap(
        () => ctx.dispatch(new Navigate([RoutesEnum.homePage])).pipe(
          tap(() => {
            ctx.patchState({
              ignorePreventLeavePopup: false,
            });
            if (isTrue(environment.reloadAfterLogout)) {
              window.location.reload();
            }
          }),
        ),
      ),
      catchError(err => {
        ctx.dispatch(new Navigate([RoutesEnum.homePage]));
        throw err;
      }),
    );
  }

  @Action(AppAction.CancelPreventExit)
  cancelPreventExit(ctx: StateContext<AppStateModel>): void {
    ctx.patchState({
      preventExit: false,
      preventExitReasonToTranslate: undefined,
    });
  }

  @Action(AppAction.PreventExit)
  preventExit(ctx: StateContext<AppStateModel>, action: AppAction.PreventExit): void {
    ctx.patchState({
      preventExit: true,
      preventExitReasonToTranslate: action.reasonToTranslate,
    });
  }

  @Action(AppAction.GetShortDoi)
  getShortDoi(ctx: StateContext<AppStateModel>, action: AppAction.GetShortDoi): Observable<any> {
    return this.httpClient.get<ShortDoi>("/api/short-doi" + urlSeparator + action.doiWrapper.longDoi, {})
      .pipe(
        tap(result => {
          ctx.dispatch(new AppAction.GetShortDoiSuccess(action, result.ShortDOI));
        }),
        catchError(error => {
          ctx.dispatch(new AppAction.GetShortDoiFail(action));
          throw error;
        }),
      );
  }

  @Action(AppAction.GetShortDoiSuccess)
  getShortDoiSuccess(ctx: StateContext<AppStateModel>, action: AppAction.GetShortDoiSuccess): void {
    let stringToCopy = action.shortDoi;
    let notificationToTranslate = MARK_AS_TRANSLATABLE("doi.notification.shortDoiCopyToClipboard.success");
    if (action.parent.doiWrapper.isFullUrl) {
      stringToCopy = environment.doiLink + stringToCopy;
      notificationToTranslate = MARK_AS_TRANSLATABLE("doi.notification.urlShortDoiCopyToClipboard.success");
    }
    ClipboardUtil.copyStringToClipboard(stringToCopy);
    this.notificationService.showInformation(notificationToTranslate, true);
  }

  @Action(AppAction.GetShortDoiFail)
  getShortDoiFail(ctx: StateContext<AppStateModel>, action: AppAction.GetShortDoiFail): void {
    let notificationToTranslate = MARK_AS_TRANSLATABLE("doi.notification.shortDoiCopyToClipboard.fail");
    if (action.parent.doiWrapper.isFullUrl) {
      notificationToTranslate = MARK_AS_TRANSLATABLE("doi.notification.urlShortDoiCopyToClipboard.fail");
    }
    this.notificationService.showWarning(notificationToTranslate);
  }

  @Action(AppAction.SetIsInTourMode)
  setIsInTourMode(ctx: StateContext<AppStateModel>, action: AppAction.SetIsInTourMode): void {
    ctx.patchState({
      isInTourMode: action.isInTourMode,
    });
  }

  @Action(AppAction.SetIsTouchInterface)
  setIsTouchInterface(ctx: StateContext<AppStateModel>, action: AppAction.SetIsTouchInterface): void {
    ctx.patchState({
      isTouchInterface: action.isTouchInterface,
    });
  }

  @Action(AppAction.DisplayCookieConsent)
  displayCookieConsent(ctx: StateContext<AppStateModel>, action: AppAction.DisplayCookieConsent): void {
    ctx.patchState({
      displayCookieConsent: action.display,
    });
  }
}
