import {SessionStorageEnum} from "@app/shared/enums/session-storage.enum";
import {Enums} from "@enums";
import {environment} from "@environments/environment";
import {OrganizationalUnit} from "@models";
import {
  isNotNullNorUndefined,
  isNullOrUndefined,
} from "solidify-frontend";

export class UserPreferencesUtil {
  static getPreferredLanguage(): Enums.Language.LanguageEnum {
    const savedLanguage = sessionStorage.getItem(SessionStorageEnum.language) as Enums.Language.LanguageEnum;
    if (isNullOrUndefined(savedLanguage) || environment.appLanguages.indexOf(savedLanguage) === -1) {
      return this.determineDefaultLanguageAndStoreIntoSessionStorage();
    }
    return savedLanguage;
  }

  private static determineDefaultLanguageAndStoreIntoSessionStorage(): Enums.Language.LanguageEnum {
    const defaultLanguageDetermined = UserPreferencesUtil.determineDefaultLanguage();
    sessionStorage.setItem(SessionStorageEnum.language, defaultLanguageDetermined);
    return defaultLanguageDetermined;
  }

  private static determineDefaultLanguage(): Enums.Language.LanguageEnum {
    const languageFromLocalBrowser = UserPreferencesUtil.findCorrespondingLanguageToLocal();
    if (languageFromLocalBrowser === null) {
      return environment.defaultLanguage;
    }
    return languageFromLocalBrowser;
  }

  private static findCorrespondingLanguageToLocal(): Enums.Language.LanguageEnum {
    const browserLanguage: string = navigator.language;

    const foundLanguage: Enums.Language.LanguageEnum = environment.appLanguages.find(l => browserLanguage.startsWith(l));
    if (isNotNullNorUndefined(foundLanguage)) {
      return foundLanguage;
    }

    return null;
  }

  static getPreferredOrgUnitInDepositMenu(listAuthorizedOrgUnit: OrganizationalUnit[]): OrganizationalUnit | undefined {
    const orgUnitId = sessionStorage.getItem(SessionStorageEnum.orgUnitForDepositMenu);
    if (isNullOrUndefined(listAuthorizedOrgUnit) || listAuthorizedOrgUnit.length === 0) {
      return undefined;
    }
    const defaultOrgUnit = listAuthorizedOrgUnit[0];
    if (isNullOrUndefined(orgUnitId)) {
      return defaultOrgUnit;
    }
    let authorizedOrgUnit = listAuthorizedOrgUnit.find(o => o.resId === orgUnitId);
    if (isNullOrUndefined(authorizedOrgUnit)) {
      authorizedOrgUnit = defaultOrgUnit;
    }
    return authorizedOrgUnit;
  }

  static setPreferredOrgUnitInDepositMenu(orgUnitId: string): string | undefined {
    sessionStorage.setItem(SessionStorageEnum.orgUnitForDepositMenu, orgUnitId);
    if (isNullOrUndefined(orgUnitId)) {
      return undefined;
    }
    return orgUnitId;
  }
}
