import {
  ArchiveMetadata,
  MetadataEnum,
} from "@app/shared/models/business/archive-metadata.model";
import {Enums} from "@enums";
import {ArchiveContributor} from "@home/models/archive.model";
import {
  isFalse,
  isNullOrUndefined,
  isTrue,
} from "solidify-frontend";
import Metadata = ArchiveMetadata.PublicMetadata;

const DATACITE_NAMESPACE: string = "http://datacite.org/schema/kernel-4";
const DLCM_URI_FINAL_ACCESS_LEVEL: string = "https://www.dlcm.ch/access-level/final";
const DLCM_URI_EMBARGO_ACCESS_LEVEL: string = "https://www.dlcm.ch/access-level/embargo";

export class MetadataUtil {
  private static readonly domParser: DOMParser = new DOMParser();

  private static parse(metadataXml: string): Document {
    return this.domParser.parseFromString(metadataXml, "text/xml");
  }

  private static prefixNSResolver(prefix: string): string | null {
    if (prefix === "d") {
      return DATACITE_NAMESPACE;
    } else {
      return null;
    }
  }

  private static getInformation(xpath: string, metadata: Metadata): string {
    const metadataXml: string = metadata[MetadataEnum.dataciteXml];
    if (isNullOrUndefined(metadataXml)) {
      console.error("metadataXml provided undefined");
      return "";
    }
    const xmlDoc = this.parse(metadataXml);
    const result = xmlDoc.evaluate(xpath, xmlDoc, this.prefixNSResolver, XPathResult.STRING_TYPE, null);
    return result.stringValue;
  }

  static getTitle(metadata: Metadata): string {
    const xpath = "/d:resource/d:titles/d:title";
    return this.getInformation(xpath, metadata);
  }

  static getOrganizationalUnitResId(metadata: Metadata): string {
    return metadata[MetadataEnum.aipOrganizationalUnit];
  }

  static getWithThumbnail(metadata: Metadata): boolean {
    const withThumbnail = metadata[MetadataEnum.aipThumbnail];
    return isTrue(withThumbnail);
  }

  static isCollection(metadata: Metadata): boolean {
    const aipUnit = metadata[MetadataEnum.aipUnit];
    return isFalse(aipUnit);
  }

  static getPublicationYear(metadata: Metadata): string {
    const xpath = "/d:resource/d:publicationYear";
    return this.getInformation(xpath, metadata);
  }

  static getAccessLevel(metadata: Metadata): string {
    const xpath = `/d:resource/d:rightsList/d:rights[@rightsURI="${DLCM_URI_FINAL_ACCESS_LEVEL}"]`;
    return this.getInformation(xpath, metadata);
  }

  static getDataSensibility(metadata: Metadata): string {
    const dataSensibility = metadata[MetadataEnum.aipDataTag];
    return isNullOrUndefined(dataSensibility) ? Enums.Deposit.DataSensitivityEnum.UNDEFINED : dataSensibility;
  }

  static getEmbargoAccessLevel(metadata: Metadata): string {
    const xpath = `/d:resource/d:rightsList/d:rights[@rightsURI="${DLCM_URI_EMBARGO_ACCESS_LEVEL}"]`;
    return this.getInformation(xpath, metadata);
  }

  static getLicenses(metadata: Metadata): string {
    const xpath = `/d:resource/d:rightsList/d:rights[@rightsURI!="${DLCM_URI_FINAL_ACCESS_LEVEL}" and @rightsURI!="${DLCM_URI_EMBARGO_ACCESS_LEVEL}"]`;
    return this.getInformation(xpath, metadata);
  }

  static getDescriptionAbstract(metadata: Metadata): string {
    const xpath = `/d:resource/d:descriptions/d:description[@descriptionType="Abstract"]`;
    return this.getInformation(xpath, metadata);
  }

  static getDescriptionOther(metadata: Metadata): string {
    const xpath = `/d:resource/d:descriptions/d:description[@descriptionType="Other"]`;
    return this.getInformation(xpath, metadata);
  }

  static getDescriptionTechnicalInfo(metadata: Metadata): string {
    const xpath = `/d:resource/d:descriptions/d:description[@descriptionType="TechnicalInfo"]`;
    return this.getInformation(xpath, metadata);
  }

  static getDescriptionTableOfContents(metadata: Metadata): string {
    const xpath = `/d:resource/d:descriptions/d:description[@descriptionType="TableOfContents"]`;
    return this.getInformation(xpath, metadata);
  }

  static getDescriptionSeriesInformation(metadata: Metadata): string {
    const xpath = `/d:resource/d:descriptions/d:description[@descriptionType="SeriesInformation"]`;
    return this.getInformation(xpath, metadata);
  }

  static getDescriptionMethods(metadata: Metadata): string {
    const xpath = `/d:resource/d:descriptions/d:description[@descriptionType="Methods"]`;
    return this.getInformation(xpath, metadata);
  }

  static getDOI(metadata: Metadata): string {
    const xpath = "/d:resource/d:identifier[@identifierType='DOI']";
    return this.getInformation(xpath, metadata);
  }

  static getContributors(metadata: Metadata): ArchiveContributor[] {
    const xpath = "/d:resource/d:creators/d:creator";
    const results = [];
    const xmlDoc = this.parse(metadata[MetadataEnum.dataciteXml]);
    const result = xmlDoc.evaluate(xpath, xmlDoc, this.prefixNSResolver, XPathResult.ORDERED_NODE_SNAPSHOT_TYPE, null);
    for (let i = 0, length = result.snapshotLength; i < length; ++i) {
      const contributor = {
        affiliation: [],
      } as ArchiveContributor;
      result.snapshotItem(i).childNodes.forEach(childNode => {
        if (childNode.nodeName === "datacite:creatorName") {
          contributor.creatorName = childNode.textContent;
        }
        if (childNode.nodeName === "datacite:givenName") {
          contributor.givenName = childNode.textContent;
        }
        if (childNode.nodeName === "datacite:familyName") {
          contributor.familyName = childNode.textContent;
        }
        if (childNode.nodeName === "datacite:affiliation") {
          contributor.affiliation.push(childNode.textContent);
        }
        if (childNode.nodeName === "datacite:nameIdentifier") {
          contributor.orcid = childNode.textContent;
        }
      });
      results.push(contributor);
    }
    return results;
  }

  static getType(metadata: Metadata): string {
    const xpath = "/d:resource/d:resourceType";
    return this.getInformation(xpath, metadata);
  }

  static getEmbargoEndDate(metadata: Metadata): Date | null {
    return this.getDate("Available", metadata);
  }

  static getAcceptedDate(metadata: Metadata): Date | null {
    return this.getDate("Accepted", metadata);
  }

  static getIssuedDate(metadata: Metadata): Date | null {
    return this.getDate("Issued", metadata);
  }

  static getSubmittedDate(metadata: Metadata): Date | null {
    return this.getDate("Submitted", metadata);
  }

  static getSize(metadata: Metadata): number | null {
    return metadata[MetadataEnum.aipSize];
  }

  static getFileNumber(metadata: Metadata): string | null {
    if (isNullOrUndefined(metadata[MetadataEnum.aipFiles])) {
      return null;
    }
    return metadata[MetadataEnum.aipFiles].toString();
  }

  private static getDate(name: string, metadata: Metadata): Date | null {
    if (isNullOrUndefined(metadata)) {
      return null;
    }
    const xpath = `/d:resource/d:dates/d:date[@dateType="${name}"]`;
    const result = this.getInformation(xpath, metadata);
    if (isNullOrUndefined(result)) {
      return null;
    }
    const date = new Date(result);
    if (date.toString() === "Invalid Date") {
      return null;
    }
    return date;
  }

  static getDatacite(metadata: Metadata): string {
    return metadata[MetadataEnum.dataciteXml];
  }
}
