import {Injectable} from "@angular/core";
import {PreIngestResourceApiEnum} from "@app/shared/enums/api.enum";
import {
  Action,
  Actions,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {ApiActionEnum} from "@shared/enums/api-action.enum";
import {ApiResourceNameEnum} from "@shared/enums/api-resource-name.enum";
import {DownloadService} from "@shared/services/download.service";
import {SharedVisualizationAction} from "@shared/stores/visualization/shared-visualization.action";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  BaseResourceState,
  BasicState,
  defaultResourceStateInitValue,
  MARK_AS_TRANSLATABLE,
  NotificationService,
  QueryParameters,
  SolidifyStateError,
} from "solidify-frontend";
import {environment} from "../../../../environments/environment";
import {LocalStateEnum} from "../../enums/local-state.enum";

export interface SharedVisualizationStateModel extends BaseResourceState {
  blob: Blob;
  blobId: number;
}

@Injectable()
@State<SharedVisualizationStateModel>({
  name: LocalStateEnum.shared_visualization,
  defaults: {
    blob: undefined,
    blobId: -1,
    ...defaultResourceStateInitValue(),
    queryParameters: new QueryParameters(environment.defaultEnumValuePageSizeOption),
  },
})
export class SharedVisualizationState extends BasicState<SharedVisualizationStateModel> {
  constructor(protected apiService: ApiService,
              protected store: Store,
              protected downloadService: DownloadService,
              protected notificationService: NotificationService,
              protected actions$: Actions) {
    super();
  }

  @Action(SharedVisualizationAction.Download)
  download(ctx: StateContext<SharedVisualizationStateModel>, action: SharedVisualizationAction.Download): Observable<Blob> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });
    const url = `${this._urlResource}/${action.data.dataFile.packageId}/${ApiResourceNameEnum.DATAFILE}/${action.data.dataFile.resId}/${ApiActionEnum.DL}`;
    return this.downloadService.downloadInMemory(url, action.data.dataFile.fileName, false)
      .pipe(
        tap(result => ctx.dispatch(new SharedVisualizationAction.DownloadSuccess(result, action.blobId))),
        catchError(error => {
          ctx.dispatch(new SharedVisualizationAction.DownloadFailed());
          throw new SolidifyStateError(error);
        }),
      );
  }

  @Action(SharedVisualizationAction.DownloadFailed)
  downloadFailed(ctx: StateContext<SharedVisualizationStateModel>, action: SharedVisualizationAction.DownloadFailed): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showError(MARK_AS_TRANSLATABLE("deposit.notification.download.fail"));
  }

  @Action(SharedVisualizationAction.DownloadSuccess)
  downloadSuccess(ctx: StateContext<SharedVisualizationStateModel>, action: SharedVisualizationAction.DownloadSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    ctx.patchState({
      blob: action.blob,
      blobId: action.blobId,
    });
  }

  protected get _urlResource(): string {
    return PreIngestResourceApiEnum.deposits;
  }
}
