import {Injectable} from "@angular/core";
import {AdminResourceApiEnum} from "@app/shared/enums/api.enum";
import {LocalStateEnum} from "@app/shared/enums/local-state.enum";
import {environment} from "@environments/environment";
import {User} from "@models";
import {
  Action,
  Actions,
  Selector,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {
  SharedUserAction,
  sharedUserActionNameSpace,
} from "@shared/stores/user/shared-user.action";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  CollectionTyped,
  defaultResourceStateInitValue,
  isFalse,
  isNullOrUndefined,
  isTrue,
  isWhiteString,
  MappingObjectUtil,
  NotificationService,
  OverrideDefaultAction,
  QueryParameters,
  QueryParametersUtil,
  ResourceState,
  ResourceStateModel,
  SolidifyStateError,
  StoreUtil,
  urlSeparator,
} from "solidify-frontend";
import {ApiActionEnum} from "@shared/enums/api-action.enum";

export interface SharedUserStateModel extends ResourceStateModel<User> {
  listPendingExternalUid: string[] | undefined;
}

@Injectable()
@State<SharedUserStateModel>({
  name: LocalStateEnum.shared_user,
  defaults: {
    ...defaultResourceStateInitValue(),
    queryParameters: new QueryParameters(environment.defaultEnumValuePageSizeOption),
    listPendingExternalUid: [],
  },
})
export class SharedUserState extends ResourceState<SharedUserStateModel, User> {
  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: sharedUserActionNameSpace,
    });
  }

  protected get _urlResource(): string {
    return AdminResourceApiEnum.users;
  }

  @Selector()
  static isLoading<T>(state: ResourceStateModel<T>): boolean {
    return StoreUtil.isLoadingState(state);
  }

  @Action(SharedUserAction.GetByExternalUid)
  getByExternalUid(ctx: StateContext<SharedUserStateModel>, action: SharedUserAction.GetByExternalUid): Observable<User> {
    let indexAlreadyExisting = -1;
    if (!isNullOrUndefined(ctx.getState().list) && isTrue(action.avoidDuplicate)) {
      indexAlreadyExisting = ctx.getState().list.findIndex(item => item.externalUid === action.externalUid);
      if (indexAlreadyExisting !== -1 && isFalse(action.replace)) {
        return;
      }
    }
    if (ctx.getState().listPendingExternalUid.find(pending => pending === action.externalUid)) {
      return;
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      listPendingExternalUid: [...(isNullOrUndefined(ctx.getState().listPendingExternalUid) ? [] : ctx.getState().listPendingExternalUid), action.externalUid],
    });
    const queryParameter = new QueryParameters(1);
    MappingObjectUtil.set(queryParameter.search.searchItems, "externalUid", action.externalUid);
    return this.apiService.get<User>(this._urlResource, queryParameter)
      .pipe(
        tap((list) => {
          if (list._data.length === 0) {
            ctx.dispatch(new SharedUserAction.GetByExternalUidFail(action));
          } else {
            ctx.dispatch(new SharedUserAction.GetByExternalUidSuccess(action, list._data[0], indexAlreadyExisting));
          }
        }),
        catchError(error => {
          ctx.dispatch(new SharedUserAction.GetByExternalUidFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(SharedUserAction.GetByExternalUidSuccess)
  getByExternalUidSuccess(ctx: StateContext<SharedUserStateModel>, action: SharedUserAction.GetByExternalUidSuccess): Observable<User> {
    if (isNullOrUndefined(action.model)) {
      return;
    }
    let list = ctx.getState().list;
    if (isNullOrUndefined(list)) {
      list = [];
    }
    if (action.indexAlreadyExisting !== -1 && isTrue(action.parentAction.replace)) {
      list = [...list];
      list[action.indexAlreadyExisting] = action.model;
    } else {
      list = [...list, action.model];
    }
    ctx.patchState({
      list: list,
      total: list.length,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      listPendingExternalUid: this.getListPendingExternalUidWithValueRemoved(ctx, action.parentAction.externalUid),
    });
  }

  @Action(SharedUserAction.GetByExternalUidFail)
  getByExternalUidFail(ctx: StateContext<SharedUserStateModel>, action: SharedUserAction.GetByExternalUidFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      listPendingExternalUid: this.getListPendingExternalUidWithValueRemoved(ctx, (action.parentAction as SharedUserAction.GetByExternalUid).externalUid),
    });
  }

  @OverrideDefaultAction()
  @Action(SharedUserAction.GetAll)
  getAll(ctx: StateContext<ResourceStateModel<User>>, action: SharedUserAction.GetAll): Observable<CollectionTyped<User>> {
    let url = this._urlResource;
    const searchItems = QueryParametersUtil.getSearchItems(action.queryParameters);
    const lastName = MappingObjectUtil.get(searchItems, "lastName");
    if (!isNullOrUndefined(lastName) && !isWhiteString(lastName)) {
      url = this._urlResource + urlSeparator + ApiActionEnum.SEARCH;
      let searchValue = "";
      lastName.trim().split(" ").forEach(term => {
        if (!isWhiteString(term)) {
          if (searchValue.length > 0) {
            searchValue += ",";
          }
          searchValue += `i-firstName~${term},i-lastName~${term}`;
        }
      });

      MappingObjectUtil.set(searchItems, "search", searchValue);
      MappingObjectUtil.delete(searchItems, "lastName");
      MappingObjectUtil.set(searchItems, "match", "any");
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      queryParameters: StoreUtil.getQueryParametersToApply(action.queryParameters, ctx),
    });
    return this.apiService.get<User>(url, ctx.getState().queryParameters)
      .pipe(
        StoreUtil.cancelUncompleted(ctx, this.actions$, [SharedUserAction.GetAll]),
        tap((collection: CollectionTyped<User>) => {
          ctx.dispatch(new SharedUserAction.GetAllSuccess(action, collection));
        }),
        catchError(error => {
          ctx.dispatch(new SharedUserAction.GetAllFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  private getListPendingExternalUidWithValueRemoved(ctx: StateContext<SharedUserStateModel>, externalUid: string): string[] {
    let listPendingExternalUid = ctx.getState().listPendingExternalUid;
    const indexOf = listPendingExternalUid.indexOf(externalUid);
    if (indexOf === -1) {
      return listPendingExternalUid;
    }
    listPendingExternalUid = [...listPendingExternalUid];
    listPendingExternalUid.splice(indexOf, 1);
    return listPendingExternalUid;
  }
}
