import {
  Actions,
  StateContext,
  Store,
} from "@ngxs/store";
import {ApiResourceNameEnum} from "@shared/enums/api-resource-name.enum";
import {ApiKeyword} from "@shared/enums/api.enum";
import {StatusHistory} from "@shared/models/status-history.model";
import {StatusHistoryNamespace} from "@shared/stores/status-history/status-history-namespace.model";
import {StatusHistoryAction} from "@shared/stores/status-history/status-history.action";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  BaseOptions,
  BaseResourceType,
  BaseState,
  BaseStateModel,
  CollectionTyped,
  defaultBaseStateInitValue,
  NotificationService,
  ObjectUtil,
  QueryParameters,
  RegisterDefaultAction,
  SolidifyStateError,
} from "solidify-frontend";

export const defaultStatusHistoryInitValue: () => StatusHistoryStateModel<BaseResourceType> = () =>
  ({
    ...defaultBaseStateInitValue(),
    history: [],
    queryParameters: new QueryParameters(),
  });

export interface StatusHistoryStateModel<TResource extends BaseResourceType> extends BaseStateModel {
  history: StatusHistory[];
  queryParameters: QueryParameters;
}

export abstract class StatusHistoryState<TStateModel extends BaseStateModel, TResource extends BaseResourceType> extends BaseState<TStateModel> {
  protected readonly _nameSpace: StatusHistoryNamespace;

  protected abstract get _urlResource(): string;

  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions,
              protected options: BaseOptions) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: options.nameSpace,
    }, StatusHistoryState);
  }

  private evaluateSubResourceUrl(parentId: string): string {
    if (parentId !== null) {
      return this._urlResource.toString().replace(`{${ApiKeyword.PARENT_ID}}`, parentId);
    } else {
      return this._urlResource.toString();
    }
  }

  @RegisterDefaultAction((statusHistoryNamespace: StatusHistoryNamespace) => statusHistoryNamespace.History)
  history(ctx: StateContext<StatusHistoryStateModel<TResource>>, action: StatusHistoryAction.History): Observable<CollectionTyped<StatusHistory>> {
    ctx.patchState({
      history: [],
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      queryParameters: this.getQueryParametersToApply(action.queryParameters, ctx),
    });
    const url = this.evaluateSubResourceUrl(action.parentId);
    return this.apiService.get<StatusHistory>(`${url}/${action.id}/${ApiResourceNameEnum.HISTORY}`, ctx.getState().queryParameters)
      .pipe(
        tap((list: CollectionTyped<StatusHistory>) => {
          ctx.dispatch(new this._nameSpace.HistorySuccess(action, list));
        }),
        catchError(error => {
          ctx.dispatch(new this._nameSpace.HistoryFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  getQueryParametersToApply(queryParameters: QueryParameters, ctx: StateContext<StatusHistoryStateModel<TResource>>): QueryParameters {
    return queryParameters == null ? ctx.getState().queryParameters : queryParameters;
  }

  updateQueryParameters<T>(ctx: StateContext<StatusHistoryStateModel<TResource>>, list: CollectionTyped<T> | null | undefined): QueryParameters {
    const queryParameters = ObjectUtil.clone(ctx.getState().queryParameters);
    const paging = ObjectUtil.clone(queryParameters.paging);
    paging.length = list === null || list === undefined ? 0 : list._page.totalItems;
    queryParameters.paging = paging;
    return queryParameters;
  }

  @RegisterDefaultAction((statusHistoryNamespace: StatusHistoryNamespace) => statusHistoryNamespace.HistorySuccess)
  historySuccess(ctx: StateContext<StatusHistoryStateModel<TResource>>, action: StatusHistoryAction.HistorySuccess): void {
    const queryParameters = this.updateQueryParameters(ctx, action.list);
    ctx.patchState({
      history: action.list._data,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      queryParameters,
    });
  }

  @RegisterDefaultAction((statusHistoryNamespace: StatusHistoryNamespace) => statusHistoryNamespace.HistoryFail)
  historyFail(ctx: StateContext<StatusHistoryStateModel<TResource>>, action: StatusHistoryAction.HistoryFail): void {
    ctx.patchState({
      history: [],
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((statusHistoryNamespace: StatusHistoryNamespace) => statusHistoryNamespace.ChangeQueryParameters)
  changeQueryParameters(ctx: StateContext<StatusHistoryStateModel<TResource>>, action: StatusHistoryAction.ChangeQueryParameters): void {
    ctx.patchState({
      queryParameters: action.queryParameters,
    });

    ctx.dispatch(new this._nameSpace.History(action.id, action.parentId, undefined));
  }
}
