import {
  HttpClient,
  HttpErrorResponse,
  HttpEventType,
  HttpHeaders,
} from "@angular/common/http";
import {UploadEventModel} from "@deposit/models/upload-event.model";
import {User} from "@models";
import {
  Actions,
  StateContext,
  Store,
} from "@ngxs/store";
import {ApiActionEnum} from "@shared/enums/api-action.enum";
import {DlcmResourceOptions} from "@shared/stores/dlcm-resource/dlcm-resource-options.model";
import {ResourceLogoActionHelper} from "@shared/stores/resource-logo/resource-logo-action.helper";
import {ResourceLogoNameSpace} from "@shared/stores/resource-logo/resource-logo-namespace.model";
import {ResourceLogoAction} from "@shared/stores/resource-logo/resource-logo.action";
import {Observable} from "rxjs";
import {
  catchError,
  map,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  BaseResourceType,
  BaseStateModel,
  defaultResourceStateInitValue,
  isNotNullNorUndefined,
  isNullOrUndefined,
  NotificationService,
  RegisterDefaultAction,
  ResourceState,
  MappingObjectUtil,
  ObjectUtil,
  StringUtil,
} from "solidify-frontend";
import {ResourceLogoStateModel} from "./resource-logo-state.model";

export const defaultResourceLogoStateInitValue: <TResourceType extends BaseResourceType> () => ResourceLogoStateModel<TResourceType> = () =>
  ({
    ...defaultResourceStateInitValue(),
    logo: undefined,
    isLoadingLogo: false,
    listLogo: {},
    isLoadingAllLogo: false,
});

export abstract class ResourceLogoState<TStateModel extends BaseStateModel, TResource extends BaseResourceType> extends ResourceState<TStateModel, TResource> {
  protected readonly _FILE_KEY: string = "file";

  protected readonly _nameSpace: ResourceLogoNameSpace;

  protected constructor(protected apiService: ApiService,
                        protected store: Store,
                        protected notificationService: NotificationService,
                        protected actions$: Actions,
                        protected options: DlcmResourceOptions,
                        protected _httpClient: HttpClient,
                        protected _mode: ResourceLogoStateModeEnum) {
    super(apiService, store, notificationService, actions$, options);
  }

  private _getDownloadAction(): ApiActionEnum {
    if (this._mode === ResourceLogoStateModeEnum.logo) {
      return ApiActionEnum.DOWNLOAD_LOGO;
    }
    if (this._mode === ResourceLogoStateModeEnum.avatar) {
      return ApiActionEnum.DOWNLOAD_AVATAR;
    }
  }

  private _getUploadAction(): ApiActionEnum {
    if (this._mode === ResourceLogoStateModeEnum.logo) {
      return ApiActionEnum.UPLOAD_LOGO;
    }
    if (this._mode === ResourceLogoStateModeEnum.avatar) {
      return ApiActionEnum.UPLOAD_AVATAR;
    }
  }

  private _getDeleteAction(): ApiActionEnum {
    if (this._mode === ResourceLogoStateModeEnum.logo) {
      return ApiActionEnum.DELETE_LOGO;
    }
    if (this._mode === ResourceLogoStateModeEnum.avatar) {
      return ApiActionEnum.DELETE_AVATAR;
    }
  }

  protected abstract get _urlLogoResource(): string;

  @RegisterDefaultAction((userAvatarNameSpace: ResourceLogoNameSpace) => userAvatarNameSpace.GetPhoto, {}, ResourceState)
  getPhoto(ctx: StateContext<ResourceLogoStateModel<TResource>>, action: ResourceLogoAction.GetPhoto): Observable<any> {
    ctx.patchState({
      isLoadingLogo: true,
    });

    let headers = new HttpHeaders();
    headers = headers.set("Content-Disposition", "attachment; filename=" + "photo");

    return this._httpClient.get(`${this._urlLogoResource}/${action.resId}/${this._getDownloadAction()}`, {
      headers,
      responseType: "blob",
    }).pipe(
      tap((data: Blob) => {
        ctx.dispatch(ResourceLogoActionHelper.getPhotoSuccess(this._nameSpace, action, data));
      }),
      catchError((error: Error | HttpErrorResponse) => {
        ctx.dispatch(ResourceLogoActionHelper.getPhotoFail(this._nameSpace, action));
        throw error;
      }),
    );
  }

  @RegisterDefaultAction((userAvatarNameSpace: ResourceLogoNameSpace) => userAvatarNameSpace.GetPhotoSuccess, {}, ResourceState)
  getPhotoSuccess(ctx: StateContext<ResourceLogoStateModel<TResource>>, action: ResourceLogoAction.GetPhotoSuccess): void {
    ctx.patchState({
      isLoadingLogo: false,
      logo: isNullOrUndefined(action.blob) ? undefined : URL.createObjectURL(action.blob),
    });
  }

  @RegisterDefaultAction((userAvatarNameSpace: ResourceLogoNameSpace) => userAvatarNameSpace.GetPhotoFail, {}, ResourceState)
  getPhotoFail(ctx: StateContext<ResourceLogoStateModel<TResource>>, action: ResourceLogoAction.GetPhotoFail): void {
    ctx.patchState({
      isLoadingLogo: false,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @RegisterDefaultAction((userAvatarNameSpace: ResourceLogoNameSpace) => userAvatarNameSpace.GetPhotoByResId, {}, ResourceState)
  getPhotoByResId(ctx: StateContext<ResourceLogoStateModel<TResource>>, action: ResourceLogoAction.GetPhotoByResId): Observable<any> {
    ctx.patchState({
      isLoadingAllLogo: true,
    });

    let headers = new HttpHeaders();
    headers = headers.set("Content-Disposition", "attachment; filename=" + "photo");

    return this._httpClient.get(`${this._urlLogoResource}/${action.resId}/${this._getDownloadAction()}`, {
      headers,
      responseType: "blob",
    }).pipe(
      tap((data: Blob) => {
        ctx.dispatch(ResourceLogoActionHelper.getPhotoByResIdSuccess(this._nameSpace, action, data));
      }),
      catchError((error: Error | HttpErrorResponse) => {
        ctx.dispatch(ResourceLogoActionHelper.getPhotoByResIdFail(this._nameSpace, action));
        throw error;
      }),
    );
  }

  @RegisterDefaultAction((userAvatarNameSpace: ResourceLogoNameSpace) => userAvatarNameSpace.GetPhotoByResIdSuccess, {}, ResourceState)
  getPhotoByResIdSuccess(ctx: StateContext<ResourceLogoStateModel<TResource>>, action: ResourceLogoAction.GetPhotoByResIdSuccess): void {
    const logoCopy = ObjectUtil.clone(ctx.getState().listLogo);
    if (isNullOrUndefined(action.blob)) {
      MappingObjectUtil.set(logoCopy, action.parentAction.resId, StringUtil.stringEmpty);
    } else {
      MappingObjectUtil.set(logoCopy, action.parentAction.resId, URL.createObjectURL(action.blob));
    }
    ctx.patchState({
      isLoadingAllLogo: false,
      listLogo: logoCopy,
    });
  }


  @RegisterDefaultAction((userAvatarNameSpace: ResourceLogoNameSpace) => userAvatarNameSpace.GetPhotoByResIdFail, {}, ResourceState)
  getPhotoByResIdFail(ctx: StateContext<ResourceLogoStateModel<TResource>>, action: ResourceLogoAction.GetPhotoByResIdFail): void {
    const logoCopy = ObjectUtil.clone(ctx.getState().listLogo);
    MappingObjectUtil.set(logoCopy, action.parentAction.resId, StringUtil.stringEmpty);

    ctx.patchState({
      isLoadingAllLogo: false,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      listLogo: logoCopy,
    });
  }

  @RegisterDefaultAction((userAvatarNameSpace: ResourceLogoNameSpace) => userAvatarNameSpace.UploadPhoto, {}, ResourceState)
  uploadPhoto(ctx: StateContext<ResourceLogoStateModel<TResource>>, action: ResourceLogoAction.UploadPhoto): Observable<any> {
    ctx.patchState({
      isLoadingLogo: true,
    });
    console.error(action.file);
    const formData = new FormData();
    formData.append(this._FILE_KEY, action.file, action.file.name);

    return this.apiService.upload(`${this._urlLogoResource}/${action.resId}/${this._getUploadAction()}`, formData)
      .pipe(
        map((event: UploadEventModel) => {
          switch (event.type) {
            case HttpEventType.DownloadProgress:
            case HttpEventType.UploadProgress:
              return;
            case HttpEventType.Response:
              if (isNotNullNorUndefined(event.body) &&
                ((this._mode === ResourceLogoStateModeEnum.avatar && isNotNullNorUndefined((event.body as User).avatar))
                  || (this._mode === ResourceLogoStateModeEnum.logo && isNotNullNorUndefined(event.body.logo)))) {
                ctx.dispatch(ResourceLogoActionHelper.uploadPhotoSuccess(this._nameSpace, action));
              } else {
                ctx.dispatch(ResourceLogoActionHelper.getPhotoFail(this._nameSpace, action));
              }
              return;
            default:
              return;
          }
        }),
        catchError((error: Error | HttpErrorResponse) => {
          ctx.dispatch(ResourceLogoActionHelper.uploadPhotoFail(this._nameSpace, action));
          throw error;
        }),
      );
  }

  @RegisterDefaultAction((userAvatarNameSpace: ResourceLogoNameSpace) => userAvatarNameSpace.UploadPhotoSuccess, {}, ResourceState)
  uploadPhotoSuccess(ctx: StateContext<ResourceLogoStateModel<TResource>>, action: ResourceLogoAction.UploadPhotoSuccess): void {
    ctx.patchState({
      isLoadingLogo: false,
    });
  }

  @RegisterDefaultAction((userAvatarNameSpace: ResourceLogoNameSpace) => userAvatarNameSpace.UploadPhotoFail, {}, ResourceState)
  uploadPhotoFail(ctx: StateContext<ResourceLogoStateModel<TResource>>, action: ResourceLogoAction.UploadPhotoFail): void {
    ctx.patchState({
      isLoadingLogo: false,
    });
  }

  @RegisterDefaultAction((userAvatarNameSpace: ResourceLogoNameSpace) => userAvatarNameSpace.DeletePhoto, {}, ResourceState)
  deletePhoto(ctx: StateContext<ResourceLogoStateModel<TResource>>, action: ResourceLogoAction.DeletePhoto): Observable<any> {
    ctx.patchState({
      isLoadingLogo: true,
    });

    return this.apiService.delete(`${this._urlLogoResource}/${action.resId}/${this._getDeleteAction()}`)
      .pipe(
        tap((result: any) => {
          ctx.dispatch(ResourceLogoActionHelper.deletePhotoSuccess(this._nameSpace, action));
        }),
        catchError((error: Error | HttpErrorResponse) => {
          ctx.dispatch(ResourceLogoActionHelper.deletePhotoSuccess(this._nameSpace, action));
          throw error;
        }),
      );
  }

  @RegisterDefaultAction((userAvatarNameSpace: ResourceLogoNameSpace) => userAvatarNameSpace.DeletePhotoSuccess, {}, ResourceState)
  deletePhotoSuccess(ctx: StateContext<ResourceLogoStateModel<TResource>>, action: ResourceLogoAction.DeletePhotoSuccess): void {
    ctx.patchState({
      isLoadingLogo: false,
      logo: undefined,
    });
  }

  @RegisterDefaultAction((userAvatarNameSpace: ResourceLogoNameSpace) => userAvatarNameSpace.DeletePhotoFail, {}, ResourceState)
  deletePhotoFail(ctx: StateContext<ResourceLogoStateModel<TResource>>, action: ResourceLogoAction.DeletePhotoFail): void {
    ctx.patchState({
      isLoadingLogo: false,
    });
  }
}

export enum ResourceLogoStateModeEnum {
  avatar,
  logo,
  dataset,
}
