import {Injectable} from "@angular/core";
import {AdminResourceApiEnum} from "@app/shared/enums/api.enum";
import {ResearchDomain} from "@models";
import {
  Action,
  Actions,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {ApiActionEnum} from "@shared/enums/api-action.enum";
import {
  SharedResearchDomainAction,
  sharedResearchDomainActionNameSpace,
} from "@shared/stores/research-domain/shared-research-domain.action";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  CollectionTyped,
  defaultResourceStateInitValue,
  isNullOrUndefined,
  isWhiteString,
  MappingObjectUtil,
  NotificationService,
  OverrideDefaultAction,
  QueryParametersUtil,
  ResourceState,
  ResourceStateModel,
  SolidifyStateError,
  StoreUtil,
  urlSeparator,
} from "solidify-frontend";
import {LocalStateEnum} from "../../enums/local-state.enum";

export interface SharedResearchDomainStateModel extends ResourceStateModel<ResearchDomain> {
  sources: string[] | undefined;
}

@Injectable()
@State<SharedResearchDomainStateModel>({
  name: LocalStateEnum.shared_researchDomain,
  defaults: {
    ...defaultResourceStateInitValue(),
    sources: undefined,
  },
})
export class SharedResearchDomainState extends ResourceState<SharedResearchDomainStateModel, ResearchDomain> {
  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: sharedResearchDomainActionNameSpace,
    });
  }

  protected get _urlResource(): string {
    return AdminResourceApiEnum.researchDomains;
  }

  @Action(SharedResearchDomainAction.GetSource)
  getSource(ctx: StateContext<SharedResearchDomainStateModel>, action: SharedResearchDomainAction.GetSource): Observable<string[]> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return this.apiService.getByIdInPath<string[]>(`${this._urlResource}/${ApiActionEnum.LIST_RESEARCH_DOMAIN_SOURCES}`)
      .pipe(
        tap(result => {
          ctx.dispatch(new SharedResearchDomainAction.GetSourceSuccess(action, result));
        }),
        catchError(error => {
          ctx.dispatch(new SharedResearchDomainAction.GetSourceFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(SharedResearchDomainAction.GetSourceSuccess)
  getSourceSuccess(ctx: StateContext<SharedResearchDomainStateModel>, action: SharedResearchDomainAction.GetSourceSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      sources: action.sources,
    });
  }

  @Action(SharedResearchDomainAction.GetSourceFail)
  getSourceFail(ctx: StateContext<SharedResearchDomainStateModel>, action: SharedResearchDomainAction.GetSourceFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @OverrideDefaultAction()
  @Action(SharedResearchDomainAction.GetAll)
  getAll(ctx: StateContext<SharedResearchDomainStateModel>, action: SharedResearchDomainAction.GetAll): Observable<CollectionTyped<ResearchDomain>> {
    let url = this._urlResource;
    const searchItems = QueryParametersUtil.getSearchItems(action.queryParameters);
    const nameOrCode = MappingObjectUtil.get(searchItems, "nameOrCode");
    if (!isNullOrUndefined(nameOrCode) && !isWhiteString(nameOrCode)) {
      MappingObjectUtil.set(searchItems, "search", `i-name~${nameOrCode},i-code~${nameOrCode}`);
      MappingObjectUtil.set(searchItems, "match", "any");
      MappingObjectUtil.delete(searchItems, "nameOrCode");
      url = this._urlResource + urlSeparator + ApiActionEnum.SEARCH;
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      queryParameters: StoreUtil.getQueryParametersToApply(action.queryParameters, ctx),
    });
    return this.apiService.get<ResearchDomain>(url, ctx.getState().queryParameters)
      .pipe(
        StoreUtil.cancelUncompleted(ctx, this.actions$, [SharedResearchDomainAction.GetAll]),
        tap((collection: CollectionTyped<ResearchDomain>) => {
          ctx.dispatch(new SharedResearchDomainAction.GetAllSuccess(action, collection));
        }),
        catchError(error => {
          ctx.dispatch(new SharedResearchDomainAction.GetAllFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }
}
