import {Injectable} from "@angular/core";
import {AdminResourceApiEnum} from "@app/shared/enums/api.enum";
import {LocalStateEnum} from "@app/shared/enums/local-state.enum";
import {
  SharedPersonAction,
  sharedPersonActionNameSpace,
} from "@app/shared/stores/person/shared-person.action";
import {environment} from "@environments/environment";
import {
  Institution,
  Person,
} from "@models";
import {
  Action,
  Actions,
  ofActionCompleted,
  Selector,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {ApiActionEnum} from "@shared/enums/api-action.enum";
import {ApiResourceNameEnum} from "@shared/enums/api-resource-name.enum";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  CollectionTyped,
  defaultResourceStateInitValue,
  isNullOrUndefined,
  isWhiteString,
  MappingObjectUtil,
  MARK_AS_TRANSLATABLE,
  NotificationService,
  ObjectUtil,
  OverrideDefaultAction,
  QueryParameters,
  QueryParametersUtil,
  ResourceState,
  ResourceStateModel,
  SolidifyStateError,
  StoreUtil, StringUtil,
  urlSeparator,
} from "solidify-frontend";
import {ActionSubActionCompletionsWrapper} from "solidify-frontend/lib/models/stores/base.action";

export interface SharedPersonStateModel extends ResourceStateModel<Person> {
  listPersonMatching: Person[];
}

@Injectable()
@State<SharedPersonStateModel>({
  name: LocalStateEnum.shared_person,
  defaults: {
    ...defaultResourceStateInitValue(),
    listPersonMatching: [],
    queryParameters: new QueryParameters(environment.defaultEnumValuePageSizeOption),
  },
})
export class SharedPersonState extends ResourceState<SharedPersonStateModel, Person> {
  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: sharedPersonActionNameSpace,
      notificationResourceCreateSuccessTextToTranslate: MARK_AS_TRANSLATABLE("shared.person.notification.resource.create"),
      notificationResourceDeleteSuccessTextToTranslate: MARK_AS_TRANSLATABLE("shared.person.notification.resource.delete"),
      notificationResourceUpdateSuccessTextToTranslate: MARK_AS_TRANSLATABLE("shared.person.notification.resource.update"),
    });
  }

  protected get _urlResource(): string {
    return AdminResourceApiEnum.people;
  }

  @Selector()
  static isLoading<T>(state: ResourceStateModel<T>): boolean {
    return StoreUtil.isLoadingState(state);
  }

  @Action(SharedPersonAction.Search)
  search(ctx: StateContext<SharedPersonStateModel>, action: SharedPersonAction.Search): Observable<boolean> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });
    return StoreUtil.dispatchSequentialActionAndWaitForSubActionsCompletion(ctx, [
      {
        action: new SharedPersonAction.SearchPerson(action.person),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(SharedPersonAction.SearchPersonSuccess)),
          this.actions$.pipe(ofActionCompleted(SharedPersonAction.SearchPersonFail)),
        ],
      },
      {
        action: new SharedPersonAction.SearchInstitutions(),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(SharedPersonAction.SearchInstitutionsSuccess)),
          this.actions$.pipe(ofActionCompleted(SharedPersonAction.SearchInstitutionsFail)),
        ],
      },
    ]).pipe(
      tap(success => {
        if (success) {
          ctx.dispatch(new SharedPersonAction.SearchSuccess(action));
        } else {
          ctx.dispatch(new SharedPersonAction.SearchFail(action));
        }
      }),
    );
  }

  @Action(SharedPersonAction.SearchSuccess)
  SearchSuccess(ctx: StateContext<SharedPersonStateModel>, action: SharedPersonAction.SearchSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(SharedPersonAction.SearchFail)
  searchFail(ctx: StateContext<SharedPersonStateModel>, action: SharedPersonAction.SearchFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(SharedPersonAction.SearchPerson)
  searchPerson(ctx: StateContext<SharedPersonStateModel>, action: SharedPersonAction.SearchPerson): Observable<CollectionTyped<Person>> {
    const map = new Map<string, string>();
    map.set("firstName", action.person.firstName);
    map.set("lastName", action.person.lastName);
    const queryParameters = new QueryParameters();
    queryParameters.search = {
      searchItems: map,
    };
    return this.apiService.get<Person>(this._urlResource, queryParameters)
      .pipe(
        tap(collection => {
          ctx.dispatch(new SharedPersonAction.SearchPersonSuccess(action, collection._data));
        }),
        catchError(error => {
          ctx.dispatch(new SharedPersonAction.SearchPersonFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(SharedPersonAction.SearchPersonSuccess)
  searchPersonSuccess(ctx: StateContext<SharedPersonStateModel>, action: SharedPersonAction.SearchPersonSuccess): void {
    ctx.patchState({
      listPersonMatching: action.list,
    });
  }

  @Action(SharedPersonAction.SearchInstitutions)
  searchInstitutions(ctx: StateContext<SharedPersonStateModel>, action: SharedPersonAction.SearchInstitutions): Observable<any> {
    const listActionSubActionCompletionsWrapper: ActionSubActionCompletionsWrapper[] = [];

    if (ctx.getState().listPersonMatching !== []) {
      ctx.getState().listPersonMatching.forEach(p => {
        listActionSubActionCompletionsWrapper.push({
          action: new SharedPersonAction.SearchPersonInstitution(p),
          subActionCompletions: [
            this.actions$.pipe(ofActionCompleted(SharedPersonAction.SearchPersonInstitutionSuccess)),
            this.actions$.pipe(ofActionCompleted(SharedPersonAction.SearchPersonInstitutionFail)),
          ],
        } as ActionSubActionCompletionsWrapper);
      });

      return StoreUtil.dispatchParallelActionAndWaitForSubActionsCompletion(ctx, listActionSubActionCompletionsWrapper)
        .pipe(
          tap(success => {
            if (success) {
              ctx.dispatch(new SharedPersonAction.SearchInstitutionsSuccess(action));
            } else {
              ctx.dispatch(new SharedPersonAction.SearchInstitutionsFail(action));
            }
          }),
        );
    }
    return ctx.dispatch(new SharedPersonAction.SearchInstitutionsSuccess(action));
  }

  @Action(SharedPersonAction.SearchPersonInstitution)
  searchPersonInstitutions(ctx: StateContext<SharedPersonStateModel>, action: SharedPersonAction.SearchPersonInstitution): Observable<CollectionTyped<Institution>> {
    return this.apiService.get<Institution>(this._urlResource + urlSeparator + action.person.resId + urlSeparator + ApiResourceNameEnum.INSTITUTION, null)
      .pipe(
        tap(collectionInstitutions => {
          ctx.dispatch(new SharedPersonAction.SearchPersonInstitutionSuccess(action, action.person, collectionInstitutions._data));
        }),
        catchError(error => {
          ctx.dispatch(new SharedPersonAction.SearchPersonInstitutionFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(SharedPersonAction.SearchPersonInstitutionSuccess)
  searchPersonInstitutionsSuccess(ctx: StateContext<SharedPersonStateModel>, action: SharedPersonAction.SearchPersonInstitutionSuccess): void {
    const listPersons: Person[] = [...ctx.getState().listPersonMatching];
    const personIndex = listPersons.findIndex(p => p.resId === action.person.resId);
    listPersons[personIndex] = ObjectUtil.clone(listPersons[personIndex]);
    listPersons[personIndex].institutions = action.institutions;
    ctx.patchState({
      listPersonMatching: listPersons,
    });
  }

  @OverrideDefaultAction()
  @Action(SharedPersonAction.GetAll)
  getAll(ctx: StateContext<ResourceStateModel<Person>>, action: SharedPersonAction.GetAll): Observable<CollectionTyped<Person>> {
    const onlyWithUser = action.onlyWithUser;
    let url = this._urlResource;
    const searchItems = QueryParametersUtil.getSearchItems(action.queryParameters);
    const fullName = MappingObjectUtil.get(searchItems, "fullName");
    if (!isNullOrUndefined(fullName) && !isWhiteString(fullName)) {
      if (onlyWithUser) {
        MappingObjectUtil.set(searchItems, "search", fullName);
      } else {
        url = this._urlResource + urlSeparator + ApiActionEnum.SEARCH;

        let searchValue = "";
        fullName.trim().split(" ").forEach(term => {
          if (!isWhiteString(term)) {
            if (searchValue.length > 0) {
              searchValue += ",";
            }
            const partOfORCIDRegex = /^\-?(\d{1,4}\-?){1,3}((\d{1,4})|(\d{1,3}(x|X)))?$/;
            if (partOfORCIDRegex.test(term)) {
              searchValue += `i-orcid~${term}`;
            } else {
              searchValue += `i-firstName~${term},i-lastName~${term}`;
            }
          }
        });

        MappingObjectUtil.set(searchItems, "search", searchValue);
        MappingObjectUtil.delete(searchItems, "fullName");
        MappingObjectUtil.set(searchItems, "match", "any");
      }
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      queryParameters: StoreUtil.getQueryParametersToApply(action.queryParameters, ctx),
    });
    if (onlyWithUser) {
      url = this._urlResource + urlSeparator + ApiActionEnum.SEARCH_WITH_USER;
    }
    return this.apiService.get<Person>(url, ctx.getState().queryParameters)
      .pipe(
        StoreUtil.cancelUncompleted(ctx, this.actions$, [SharedPersonAction.GetAll]),
        tap((collection: CollectionTyped<Person>) => {
          ctx.dispatch(new SharedPersonAction.GetAllSuccess(action, collection));
        }),
        catchError(error => {
          ctx.dispatch(new SharedPersonAction.GetAllFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }
}
