import {
  Actions,
  StateContext,
  Store,
} from "@ngxs/store";
import {ApiResourceNameEnum} from "@shared/enums/api-resource-name.enum";
import {StatusHistory} from "@shared/models/status-history.model";
import {DlcmResourceOptions} from "@shared/stores/dlcm-resource/dlcm-resource-options.model";
import {DlcmResourceAction} from "@shared/stores/dlcm-resource/dlcm-resource.action";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  BaseResourceType,
  CollectionTyped,
  defaultResourceStateInitValue,
  NotificationService,
  RegisterDefaultAction,
  ResourceState,
  SolidifyStateError,
} from "solidify-frontend";
import {BaseStateModel} from "solidify-frontend/lib/models";
import {DlcmResourceActionHelper} from "./dlcm-resource-action.helper";
import {DlcmResourceNameSpace} from "./dlcm-resource-namespace.model";
import {DlcmResourceStateModel} from "./dlcm-resource-state.model";

export const defaultDlcmResourceStateInitValue: DlcmResourceStateModel<BaseResourceType> = {
  ...defaultResourceStateInitValue(),
  history: [],
  isLoadingHistoryCounter: 0,
};

export abstract class DlcmResourceState<TResource extends BaseResourceType> extends ResourceState<BaseStateModel, TResource> {
  protected readonly _nameSpace: DlcmResourceNameSpace;

  protected constructor(protected apiService: ApiService,
                        protected store: Store,
                        protected notificationService: NotificationService,
                        protected actions$: Actions,
                        protected options: DlcmResourceOptions) {
    super(apiService, store, notificationService, actions$, options);
  }

  @RegisterDefaultAction((dlcmResourceNameSpace: DlcmResourceNameSpace) => dlcmResourceNameSpace.History, {}, ResourceState)
  history(ctx: StateContext<DlcmResourceStateModel<TResource>>, action: DlcmResourceAction.History): Observable<CollectionTyped<StatusHistory>> {
    ctx.patchState({
      isLoadingHistoryCounter: ctx.getState().isLoadingHistoryCounter + 1,
    });
    return this.apiService.get<StatusHistory>(`${this._urlResource}/${action.id}/${ApiResourceNameEnum.HISTORY}`, null)
      .pipe(
        tap((list: CollectionTyped<StatusHistory>) => {
          ctx.dispatch(DlcmResourceActionHelper.historySuccess(this._nameSpace, action, list));
        }),
        catchError(error => {
          ctx.dispatch(DlcmResourceActionHelper.historyFail(this._nameSpace, action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @RegisterDefaultAction((dlcmResourceNameSpace: DlcmResourceNameSpace) => dlcmResourceNameSpace.HistorySuccess, {}, ResourceState)
  historySuccess(ctx: StateContext<DlcmResourceStateModel<TResource>>, action: DlcmResourceAction.HistorySuccess): void {
    ctx.patchState({
      history: action.list._data,
      isLoadingHistoryCounter: ctx.getState().isLoadingHistoryCounter - 1,
    });
  }
}
