import {
  HttpClient,
  HttpHeaders,
} from "@angular/common/http";
import {Injectable} from "@angular/core";
import {AccessResourceApiEnum} from "@app/shared/enums/api.enum";
import {LocalStateEnum} from "@app/shared/enums/local-state.enum";
import {urlSeparator} from "@app/shared/enums/routes.enum";
import {Enums} from "@enums";
import {environment} from "@environments/environment";
import {Archive} from "@home/models/archive.model";
import {Metadata} from "@models";
import {Navigate} from "@ngxs/router-plugin";
import {
  Action,
  Actions,
  Selector,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {ApiActionEnum} from "@shared/enums/api-action.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {ArchiveHelper} from "@shared/helpers/archive.helper";
import {DownloadService} from "@shared/services/download.service";
import {
  SharedArchiveAction,
  sharedArchiveActionNameSpace,
} from "@shared/stores/archive/shared-archive.action";
import {
  interval,
  Observable,
  of,
  pipe,
} from "rxjs";
import {
  catchError,
  filter,
  map,
  mergeMap,
  startWith,
  takeWhile,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  CollectionTyped,
  defaultResourceStateInitValue,
  ErrorDto,
  HttpStatus,
  isFalse,
  isNonEmptyString,
  isNullOrUndefined,
  isTrue,
  MappingObjectUtil,
  MARK_AS_TRANSLATABLE,
  MemoizedUtil,
  NotificationService,
  OverrideDefaultAction,
  QueryParameters,
  QueryParametersUtil,
  ResourceState,
  ResourceStateModel,
  SolidifyStateError,
  StoreUtil,
  StringUtil,
} from "solidify-frontend";

export const PARAM_QUERY_ORGUNIT: string = "queryOrganizationalUnit";
export const PARAM_QUERY_SEARCH: string = "querySearch";
const DEFAULT_SEARCH: string = "*";

export interface SharedArchiveStateModel extends ResourceStateModel<Archive> {
  isPooling: string[];
  isPendingRequest: string[];
  isFirstTimeError: string[];
  listRelativeArchive: Archive[];
}

@Injectable()
@State<SharedArchiveStateModel>({
  name: LocalStateEnum.shared_archive,
  defaults: {
    ...defaultResourceStateInitValue(),
    isPooling: [],
    isPendingRequest: [],
    isFirstTimeError: [],
    listRelativeArchive: undefined,
    queryParameters: new QueryParameters(environment.defaultEnumValuePageSizeOption),
  },
  children: [],
})
export class SharedArchiveState extends ResourceState<SharedArchiveStateModel, Archive> {
  constructor(protected apiService: ApiService,
              protected store: Store,
              protected httpClient: HttpClient,
              protected notificationService: NotificationService,
              protected actions$: Actions,
              private downloadService: DownloadService) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: sharedArchiveActionNameSpace,
    });
  }

  private readonly PATH_FILTER: string = "query";

  protected get _urlResource(): string {
    return AccessResourceApiEnum.publicMetadata;
  }

  private readonly IS_POOLING_KEY: keyof SharedArchiveStateModel = "isPooling";
  private readonly IS_PENDING_REQUEST_KEY: keyof SharedArchiveStateModel = "isPendingRequest";
  private readonly IS_FIRST_TIME_ERROR_KEY: keyof SharedArchiveStateModel = "isFirstTimeError";

  @Selector()
  static isLoading(state: SharedArchiveStateModel): boolean {
    return StoreUtil.isLoadingState(state);
  }

  static isDownloadPreparation(store: Store, archiveResId: string): Observable<boolean> {
    return MemoizedUtil.select(store, SharedArchiveState, state => state.isPooling).pipe(
      map(list => list.includes(archiveResId)),
    );
  }

  @OverrideDefaultAction()
  @Action(SharedArchiveAction.GetAll)
  getAll(ctx: StateContext<ResourceStateModel<Archive>>, action: SharedArchiveAction.GetAll): Observable<CollectionTyped<Archive>> {
    let reset = {};
    if (!action.keepCurrentContext) {
      reset = {
        list: undefined,
        total: 0,
      };
    }

    let queryParameters = StoreUtil.getQueryParametersToApply(action.queryParameters, ctx);
    queryParameters = QueryParametersUtil.clone(queryParameters);
    let searchValue = DEFAULT_SEARCH;
    const search = MappingObjectUtil.get(queryParameters.search.searchItems, PARAM_QUERY_SEARCH);
    if (isNonEmptyString(search)) {
      searchValue = search;
    }
    const orgUnitParam = MappingObjectUtil.get(queryParameters.search.searchItems, PARAM_QUERY_ORGUNIT);
    // let facetsSelected: SearchFacet[] = [];
    if (isNonEmptyString(orgUnitParam)) {
      searchValue = searchValue + " AND " + "aip-organizational-unit:" + orgUnitParam;
      // facetsSelected = [{
      //   field: "organizational-units-id",
      //   equalValue: orgUnitParam,
      //   type: undefined,
      // }];
    }

    MappingObjectUtil.set(queryParameters.search.searchItems, this.PATH_FILTER, searchValue);
    MappingObjectUtil.delete(queryParameters.search.searchItems, PARAM_QUERY_SEARCH);
    MappingObjectUtil.delete(queryParameters.search.searchItems, PARAM_QUERY_ORGUNIT);
    queryParameters.paging.pageSize = 5;
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      queryParameters: queryParameters,
      ...reset,
    });

    // const json = JSON.stringify(facetsSelected);
    // let headers: HttpHeaders = new HttpHeaders();
    // headers = headers.set("Content-Type", "application/x-www-form-urlencoded");
    // headers = headers.set("Accept", "application/json");
    // const queryParametersHttp = ApiService.getQueryParameters(undefined, queryParameters);
    // let body = new HttpParams();
    // body = body.set("conditions", json);
    // return this.httpClient.post<CollectionTyped<ArchiveMetadata>>(AccessResourceApiEnum.publicMetadata + urlSeparator + ApiActionEnum.SEARCH,
    //   body,
    //   {
    //     headers,
    //     params: queryParametersHttp,
    //   },
    // ).pipe(
    return this.apiService.get<Metadata>(this._urlResource + urlSeparator + ApiActionEnum.SEARCH, queryParameters)
      .pipe(
        action.cancelIncomplete ? StoreUtil.cancelUncompleted(ctx, this.actions$, [this._nameSpace.GetAll, Navigate]) : pipe(),
        map((collection: CollectionTyped<Metadata>) => {
          // TODO ADD ON SOLIDIFY METHOD THAT ALLOW TO TRANSFORM RESULT RETURNED FROM BACKEND
          const collectionArchive = {
            _links: collection._links,
            _page: collection._page,
            _facets: collection._facets,
          } as CollectionTyped<Archive>;
          collectionArchive._data = ArchiveHelper.adaptListArchivesMetadataInArchive(collection._data);
          ctx.dispatch(new SharedArchiveAction.GetAllSuccess(action, collectionArchive));
          return collectionArchive;
        }),
        catchError(error => {
          ctx.dispatch(new SharedArchiveAction.GetAllFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @OverrideDefaultAction()
  @Action(SharedArchiveAction.LoadNextChunkList)
  loadNextChunkList(ctx: StateContext<SharedArchiveStateModel>, action: SharedArchiveAction.LoadNextChunkList): Observable<CollectionTyped<Archive>> {
    const queryParameters = QueryParametersUtil.clone(ctx.getState().queryParameters);
    if (!StoreUtil.isNextChunkAvailable(queryParameters)) {
      return;
    }

    queryParameters.paging.pageIndex = queryParameters.paging.pageIndex + 1;
    ctx.patchState({
      isLoadingChunk: true,
      queryParameters: queryParameters,
    });

    return this.apiService.get<Metadata>(this._urlResource + urlSeparator + ApiActionEnum.SEARCH, ctx.getState().queryParameters)
      .pipe(
        StoreUtil.cancelUncompleted(ctx, this.actions$, [this._nameSpace.GetAll, Navigate]),
        map((collection: CollectionTyped<Metadata>) => {
          // TODO ADD ON SOLIDIFY METHOD THAT ALLOW TO TRANSFORM RESULT RETURNED FROM BACKEND
          const collectionArchive = {
            _links: collection._links,
            _page: collection._page,
            _facets: collection._facets,
          } as CollectionTyped<Archive>;
          collectionArchive._data = ArchiveHelper.adaptListArchivesMetadataInArchive(collection._data);
          ctx.dispatch(new SharedArchiveAction.LoadNextChunkListSuccess(action, collectionArchive));
          return collectionArchive;
        }),
        catchError(error => {
          ctx.dispatch(new SharedArchiveAction.LoadNextChunkListFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @OverrideDefaultAction()
  @Action(SharedArchiveAction.GetById)
  getById(ctx: StateContext<SharedArchiveStateModel>, action: SharedArchiveAction.GetById): Observable<any> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      current: undefined,
    });

    return this.apiService.getById<Metadata>(this._urlResource, action.id)
      .pipe(
        tap(model => ctx.dispatch(new SharedArchiveAction.GetByIdSuccess(action, ArchiveHelper.adaptArchiveMetadataInArchive(model)))),
        catchError(error => {
          ctx.dispatch(new SharedArchiveAction.GetByIdFail(action));
          throw error;
        }),
      );
  }

  @OverrideDefaultAction()
  @Action(SharedArchiveAction.AddInListById)
  addInListById(ctx: StateContext<SharedArchiveStateModel>, action: SharedArchiveAction.AddInListById): Observable<Archive> {
    let indexAlreadyExisting = -1;
    if (!isNullOrUndefined(ctx.getState().list) && isTrue(action.avoidDuplicate)) {
      indexAlreadyExisting = ctx.getState().list.findIndex(item => item.resId === action.resId);
      if (indexAlreadyExisting !== -1 && isFalse(action.replace)) {
        return;
      }
    }
    if (ctx.getState().listPendingResId.find(pending => pending === action.resId)) {
      return;
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      listPendingResId: [...(isNullOrUndefined(ctx.getState().listPendingResId) ? [] : ctx.getState().listPendingResId), action.resId],
    });
    return this.apiService.getById<Metadata>(this._urlResource, action.resId)
      .pipe(
        map((model: Metadata) => {
          const archive = ArchiveHelper.adaptArchiveMetadataInArchive(model);
          ctx.dispatch(new SharedArchiveAction.AddInListByIdSuccess(action, archive, indexAlreadyExisting));
          return archive;
        }),
        catchError(error => {
          ctx.dispatch(new SharedArchiveAction.AddInListByIdFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(SharedArchiveAction.Download)
  download(ctx: StateContext<SharedArchiveStateModel>, action: SharedArchiveAction.Download): any {
    this.notificationService.showInformation(MARK_AS_TRANSLATABLE("home.archive.browsing.download.notification.inPreparation"));

    this._setIsPooling(ctx, action, true);
    this._setIsPendingRequest(ctx, action, false);
    this._setIsFirstTimeError(ctx, action, false);

    return interval(2000)
      .pipe(
        takeWhile(() => this._getIsPooling(ctx, action)),
        filter(() => !this._getIsPendingRequest(ctx, action)),
        startWith(0),
        mergeMap(() => {
          this._setIsPendingRequest(ctx, action, true);
          return this._getDownloadStatus(ctx, action.archive.resId).pipe(
            tap(() => this._setIsPendingRequest(ctx, action, false)),
          );
        }),
        tap((res: string) => {
          if (res === StringUtil.stringEmpty) {
            this._setIsPendingRequest(ctx, action, true);
            this._prepareDownload(ctx, action.archive.resId).pipe(
              tap(() => {
                this._setIsPendingRequest(ctx, action, false);
              }),
              catchError((e: ErrorDto) => {
                if (e.status === HttpStatus.FORBIDDEN) {
                  this._setIsPendingRequest(ctx, action, false);
                  this._setIsPooling(ctx, action, false);
                  ctx.dispatch(new SharedArchiveAction.DownloadFail(action, LabelTranslateEnum.fileDownloadForbidden));
                  return;
                }
                return of(StringUtil.stringEmpty);
              }),
            ).subscribe();
            return;
          }
          if (res === Enums.Order.StatusEnum.READY) {
            this._setIsPooling(ctx, action, false);
            ctx.dispatch(new SharedArchiveAction.DownloadStart(action));
            return;
          }
          if (res === Enums.Order.StatusEnum.INERROR) {
            this._setIsPooling(ctx, action, false);
            if (this._getIsFirstTimeError(ctx, action) === false) {
              ctx.dispatch(new SharedArchiveAction.DownloadFail(action));
              this._setIsFirstTimeError(ctx, action, true);
            }
            return;
          }
        }),
        catchError(error => {
          this._setIsPooling(ctx, action, false);
          throw error;
        }),
      );
  }

  private _getDownloadStatus(ctx: StateContext<SharedArchiveStateModel>, resId: string): Observable<string> {
    let headers = new HttpHeaders();
    headers = headers.set("Content-Type", "text/plain;charset=UTF-8");

    return this.httpClient.get(this._urlResource + urlSeparator + resId
      + urlSeparator + ApiActionEnum.DOWNLOAD_STATUS, {
      headers,
      responseType: "text",
    }).pipe(
      catchError((e: ErrorDto) => {
        if (e.status === HttpStatus.INTERNAL_SERVER_ERROR) {
          return of(Enums.Order.StatusEnum.INERROR);
        }
        return of(StringUtil.stringEmpty);
      }),
    );
  }

  private _prepareDownload(ctx: StateContext<SharedArchiveStateModel>, resId: string): Observable<any> {
    let headers = new HttpHeaders();
    headers = headers.set("Content-Type", "text/plain;charset=UTF-8");
    return this.httpClient.post(this._urlResource + urlSeparator + resId
      + urlSeparator + ApiActionEnum.PREPARE_DOWNLOAD, {
      headers,
      responseType: "text",
    });
  }

  @Action(SharedArchiveAction.DownloadStart)
  downloadStart(ctx: StateContext<SharedArchiveStateModel>, action: SharedArchiveAction.DownloadStart): void {
    const archive = (action.parentAction as SharedArchiveAction.Download).archive;
    const isPublic = archive.accessLevel === Enums.Deposit.AccessEnum.PUBLIC;
    const fileName = archive.title + ".zip";
    const fileSize = archive.size;
    this.downloadService.download(isPublic, `${this._urlResource}/${archive.resId}/${ApiActionEnum.DL}`, fileName, fileSize, archive.resId);
  }

  @Action(SharedArchiveAction.DownloadFail)
  downloadFail(ctx: StateContext<SharedArchiveStateModel>, action: SharedArchiveAction.DownloadFail): void {
    this.notificationService.showError(action.customMessageToTranslate ?? LabelTranslateEnum.fileDownloadFail);
  }

  private _getIsPooling(ctx: StateContext<SharedArchiveStateModel>, action: SharedArchiveAction.Download): boolean {
    return this._isIncludedInList(ctx, action, this.IS_POOLING_KEY);
  }

  private _setIsPooling(ctx: StateContext<SharedArchiveStateModel>, action: SharedArchiveAction.Download, toAdd: boolean): void {
    return this._setInList(ctx, action, this.IS_POOLING_KEY, toAdd);
  }

  private _getIsPendingRequest(ctx: StateContext<SharedArchiveStateModel>, action: SharedArchiveAction.Download): boolean {
    return this._isIncludedInList(ctx, action, this.IS_PENDING_REQUEST_KEY);
  }

  private _setIsPendingRequest(ctx: StateContext<SharedArchiveStateModel>, action: SharedArchiveAction.Download, toAdd: boolean): void {
    return this._setInList(ctx, action, this.IS_PENDING_REQUEST_KEY, toAdd);
  }

  private _getIsFirstTimeError(ctx: StateContext<SharedArchiveStateModel>, action: SharedArchiveAction.Download): boolean {
    return this._isIncludedInList(ctx, action, this.IS_FIRST_TIME_ERROR_KEY);
  }

  private _setIsFirstTimeError(ctx: StateContext<SharedArchiveStateModel>, action: SharedArchiveAction.Download, toAdd: boolean): void {
    return this._setInList(ctx, action, this.IS_FIRST_TIME_ERROR_KEY, toAdd);
  }

  private _isIncludedInList(ctx: StateContext<SharedArchiveStateModel>, action: SharedArchiveAction.Download, key: keyof SharedArchiveStateModel): boolean {
    return (ctx.getState()[key] as string[]).includes(action.archive.resId);
  }

  private _setInList(ctx: StateContext<SharedArchiveStateModel>, action: SharedArchiveAction.Download, key: keyof SharedArchiveStateModel, toAdd: boolean): void {
    if (toAdd) {
      this._addInList(ctx, action, key);
    } else {
      this._removeInList(ctx, action, key);
    }
  }

  private _addInList(ctx: StateContext<SharedArchiveStateModel>, action: SharedArchiveAction.Download, key: keyof SharedArchiveStateModel): void {
    const list = ctx.getState()[key] as string[];
    if (!list.includes(action.archive.resId)) {
      ctx.patchState({
        [key]: [...list, action.archive.resId],
      });
    }
  }

  private _removeInList(ctx: StateContext<SharedArchiveStateModel>, action: SharedArchiveAction.Download, key: keyof SharedArchiveStateModel): void {
    const list = ctx.getState()[key] as string[];
    const archiveId = action.archive.resId;
    const indexOf = list.indexOf(archiveId);
    if (indexOf !== -1) {
      const listPending = [...list];
      listPending.splice(indexOf, 1);
      ctx.patchState({
        [key]: listPending,
      });
    }
  }
}

