import {Injectable} from "@angular/core";
import {AppState} from "@app/stores/app.state";
import {AppArchiveAclState} from "@app/stores/archive-acl/app-archive-acl.state";
import {DepositAction} from "@deposit/stores/deposit.action";
import {DepositState} from "@deposit/stores/deposit.state";
import {Enums} from "@enums";
import {
  Deposit,
  Role,
} from "@models";
import {
  Actions,
  Store,
} from "@ngxs/store";
import {
  Observable,
  of,
} from "rxjs";
import {map} from "rxjs/operators";
import {
  isFalse,
  isNotNullNorUndefined,
  isNullOrUndefined,
  MemoizedUtil,
} from "solidify-frontend";

@Injectable({
  providedIn: "root",
})
export class SecurityService {

  private readonly ORGUNIT_ROLE_NEED_ACCESS_TO_CLOSED_DATASET: Enums.Role.RoleEnum[] = [Enums.Role.RoleEnum.STEWARD, Enums.Role.RoleEnum.MANAGER];
  private readonly ORGUNIT_ROLE_NEED_TO_EDIT: Enums.Role.RoleEnum[] = [Enums.Role.RoleEnum.MANAGER];
  private readonly DEPOSIT_ROLE_NEED_TO_CREATE: Enums.Role.RoleEnum[] = [Enums.Role.RoleEnum.MANAGER, Enums.Role.RoleEnum.STEWARD, Enums.Role.RoleEnum.APPROVER, Enums.Role.RoleEnum.CREATOR];
  private readonly DEPOSIT_ROLE_NEED_TO_EDIT: Enums.Role.RoleEnum[] = [Enums.Role.RoleEnum.MANAGER, Enums.Role.RoleEnum.STEWARD, Enums.Role.RoleEnum.APPROVER, Enums.Role.RoleEnum.CREATOR];
  private readonly DEPOSIT_ROLE_NEED_TO_VALIDATE: Enums.Role.RoleEnum[] = [Enums.Role.RoleEnum.MANAGER, Enums.Role.RoleEnum.STEWARD, Enums.Role.RoleEnum.APPROVER];
  private readonly DEPOSIT_STATUS_ALLOW_ALTERATION: Enums.Deposit.StatusEnum[] = [Enums.Deposit.StatusEnum.INPROGRESS];
  private readonly DEPOSIT_STATUS_ALLOW_VALIDATION: Enums.Deposit.StatusEnum[] = [Enums.Deposit.StatusEnum.INVALIDATION];

  constructor(private readonly _store: Store,
              private readonly _actions$: Actions) {
  }

  canCreateDepositOnOrgUnit(orgUnitId: string): boolean {
    if (this.isRootOrAdmin()) {
      return true;
    }
    if (!this.isMemberOfOrgUnit(orgUnitId)) {
      return false;
    }
    return this.currentUserHaveRoleInListForOrgUnit(orgUnitId, this.DEPOSIT_ROLE_NEED_TO_CREATE);
  }

  isManagerOfOrgUnit(orgUnitId: string): boolean {
    if (this.isRootOrAdmin()) {
      return true;
    }
    if (!this.isMemberOfOrgUnit(orgUnitId)) {
      return false;
    }
    return this.currentUserHaveRoleInListForOrgUnit(orgUnitId, this.ORGUNIT_ROLE_NEED_TO_EDIT);
  }

  isStewardOfOrgUnit(orgUnitId: string): boolean {
    if (this.isRootOrAdmin()) {
      return true;
    }
    if (!this.isMemberOfOrgUnit(orgUnitId)) {
      return false;
    }
    return this.currentUserHaveRoleInListForOrgUnit(orgUnitId, this.ORGUNIT_ROLE_NEED_ACCESS_TO_CLOSED_DATASET);
  }

  isActiveArchiveAcl(archiveId: string): boolean {
    const listAcl = MemoizedUtil.listSnapshot(this._store, AppArchiveAclState);
    const archiveAcl = listAcl.find(a => a.aipId === archiveId);
    return isNotNullNorUndefined(archiveAcl) && isFalse(archiveAcl.expired);
  }

  canSeeDetailDeposit(deposit: Deposit): boolean {
    if (this.isRootOrAdmin()) {
      return true;
    }
    return this.isMemberOfOrgUnit(deposit.organizationalUnitId);
  }

  canSeeDetailDepositById(depositId: string): Observable<boolean> {
    if (this.isRootOrAdmin()) {
      return of(true);
    }

    let currentDeposit = MemoizedUtil.currentSnapshot(this._store, DepositState);
    if (isNullOrUndefined(currentDeposit) || currentDeposit.resId !== depositId) {
      return this._store.dispatch(new DepositAction.GetById(depositId)).pipe(
        map(state => {
          currentDeposit = MemoizedUtil.currentSnapshot(this._store, DepositState);
          return this.isMemberOfOrgUnit(currentDeposit.organizationalUnitId);
        }),
      );
    } else {
      return of(this.isMemberOfOrgUnit(currentDeposit.organizationalUnitId));
    }
  }

  canEditDeposit(deposit: Deposit, currentRoleOnOrgUnit: string[]): boolean {
    if (isFalse(this.depositInEditionStep(deposit))) {
      return false;
    }
    const canSeeDetail = this.canSeeDetailDeposit(deposit);
    if (isFalse(canSeeDetail)) {
      return false;
    }
    return this.isCurrentRoleIdOnOrgUnitAuthorized(currentRoleOnOrgUnit, this.DEPOSIT_ROLE_NEED_TO_EDIT);
  }

  canApproveOrRejectDeposit(deposit: Deposit, currentRoleOnOrgUnit: string[]): boolean {
    if (isFalse(this.depositInValidationStep(deposit))) {
      return false;
    }
    const canSeeDetail = this.canSeeDetailDeposit(deposit);
    if (isFalse(canSeeDetail)) {
      return false;
    }
    return this.isCurrentRoleIdOnOrgUnitAuthorized(currentRoleOnOrgUnit, this.DEPOSIT_ROLE_NEED_TO_VALIDATE);
  }

  depositInValidationStep(deposit: Deposit): boolean {
    return this.depositInState(deposit, this.DEPOSIT_STATUS_ALLOW_VALIDATION);
  }

  depositInEditionStep(deposit: Deposit): boolean {
    return this.depositInState(deposit, this.DEPOSIT_STATUS_ALLOW_ALTERATION);
  }

  private depositInState(deposit: Deposit, inState: Enums.Deposit.StatusEnum[]): boolean {
    return inState.includes(deposit.status);
  }

  private isCurrentRoleOnOrgUnitAuthorized(currentRoleOnOrgUnit: Role, listRolesAuthorized: Enums.Role.RoleEnum[]): boolean {
    return listRolesAuthorized.includes(currentRoleOnOrgUnit.resId as Enums.Role.RoleEnum);
  }

  private isCurrentRoleIdOnOrgUnitAuthorized(currentRoleOnOrgUnit: string[], listRolesAuthorized: Enums.Role.RoleEnum[]): boolean {
    return currentRoleOnOrgUnit.some(currentRole => listRolesAuthorized.includes(currentRole as Enums.Role.RoleEnum));
  }

  canEditDepositById(depositId: string): Observable<boolean> {
    if (this.isRootOrAdmin()) {
      return of(true);
    }

    return this.canSeeDetailDepositById(depositId).pipe(
      map((isAuthorized: boolean) => {
        if (isFalse(isAuthorized)) {
          return false;
        }
        const currentDeposit = MemoizedUtil.currentSnapshot(this._store, DepositState);
        if (!this.depositInEditionStep(currentDeposit)) {
          return false;
        }
        return this.currentUserHaveRoleInListForOrgUnit(currentDeposit.organizationalUnitId, this.DEPOSIT_ROLE_NEED_TO_EDIT);
      }),
    );
  }

  private currentUserHaveRoleInListForOrgUnit(orgUnitId: string, listRolesAuthorized: Enums.Role.RoleEnum[]): boolean {
    const currentRoleInOrgUnit = this.getRoleInOrgUnit(orgUnitId);
    if (isNullOrUndefined(currentRoleInOrgUnit)) {
      return false;
    }
    return this.isCurrentRoleOnOrgUnitAuthorized(currentRoleInOrgUnit, listRolesAuthorized);
  }

  isRootOrAdmin(): boolean {
    const applicationRoles = this._store.selectSnapshot(AppState.currentUserApplicationRoleResId);
    if (isNullOrUndefined(applicationRoles)) {
      return false;
    }
    return applicationRoles.includes(Enums.UserApplicationRole.UserApplicationRoleEnum.admin) || applicationRoles.includes(Enums.UserApplicationRole.UserApplicationRoleEnum.root);
  }

  isRoot(): boolean {
    const applicationRoles = this._store.selectSnapshot(AppState.currentUserApplicationRoleResId);
    if (isNullOrUndefined(applicationRoles)) {
      return false;
    }
    return applicationRoles.includes(Enums.UserApplicationRole.UserApplicationRoleEnum.root);
  }

  isUser(): boolean {
    const applicationRoles = this._store.selectSnapshot(AppState.currentUserApplicationRoleResId);
    if (isNullOrUndefined(applicationRoles)) {
      return false;
    }
    return applicationRoles.includes(Enums.UserApplicationRole.UserApplicationRoleEnum.user);
  }


  getRoleInOrgUnit(orgUnitId: string): Role | undefined {
    const listAuthorizedOrgUnit = this._store.selectSnapshot(AppState.listAuthorizedOrganizationalUnit);
    return listAuthorizedOrgUnit.find(o => isNotNullNorUndefined(o.role) && o.resId === orgUnitId)?.role;
  }

  getRoleEnumInOrgUnit(orgUnitId: string): Enums.Role.RoleEnum | undefined {
    return this.getRoleInOrgUnit(orgUnitId)?.resId as Enums.Role.RoleEnum;
  }

  isMemberOfOrgUnit(orgUnitId: string): boolean {
    return isNotNullNorUndefined(this.getRoleInOrgUnit(orgUnitId));
  }

  isLoggedIn(): boolean {
    return MemoizedUtil.selectSnapshot(this._store, AppState, state => state.isLoggedIn);
  }

  isLoggedInObs(): Observable<boolean> {
    return MemoizedUtil.select(this._store, AppState, state => state.isLoggedIn);
  }
}
