import {DOCUMENT} from "@angular/common";
import {
  Inject,
  Injectable,
  Renderer2,
} from "@angular/core";
import {
  NavigationEnd,
  Router,
} from "@angular/router";
import {AppAction} from "@app/stores/app.action";
import {environment} from "@environments/environment";
import {Store} from "@ngxs/store";
import {LocalStorageEnum} from "@shared/enums/local-storage.enum";
import {LocalStorageHelper} from "@shared/helpers/local-storage.helper";
import {CookieService} from "ngx-cookie";
import {tap} from "rxjs/operators";
import {isNullOrUndefined} from "solidify-frontend";

declare let gtag: Function;

@Injectable({
  providedIn: "root",
})
export class GoogleAnalyticsService {
  private static readonly _KEY_CONFIG: string = "config";
  private readonly _ATTRIBUTE_SRC: string = "src";
  private readonly _ATTRIBUTE_SRC_GA_DIFFER_KEY: string = "src-ga-differ";

  constructor(private readonly _store: Store,
              private readonly _cookieService: CookieService,
              @Inject(DOCUMENT) private readonly _document: Document,
              private router: Router) {
  }

  init(renderer: Renderer2): void {
    const idGoogleAnalytics = environment.googleAnalyticsId;
    if (isNullOrUndefined(idGoogleAnalytics)) {
      return;
    }

    const isCookieAuthorized = LocalStorageHelper.getItem(LocalStorageEnum.cookieAuthorized);
    if (isCookieAuthorized !== "true") {
      GoogleAnalyticsService._setConfigGoogleAnalyticsStoreCookie(false);

      if (isNullOrUndefined(isCookieAuthorized)) {
        this._store.dispatch(new AppAction.DisplayCookieConsent(true));
      }
    }

    gtag(GoogleAnalyticsService._KEY_CONFIG, idGoogleAnalytics, {anonymize_ip: true});
    const script = this._document.querySelectorAll(`script[${this._ATTRIBUTE_SRC_GA_DIFFER_KEY}]`)[0];
    const src = script.attributes[this._ATTRIBUTE_SRC_GA_DIFFER_KEY].textContent;
    renderer.setAttribute(script, this._ATTRIBUTE_SRC, src + idGoogleAnalytics);
    renderer.removeAttribute(script, this._ATTRIBUTE_SRC_GA_DIFFER_KEY);

    this.router.events.pipe(
      tap(event => {
        if (event instanceof NavigationEnd) {
          gtag(GoogleAnalyticsService._KEY_CONFIG, idGoogleAnalytics,
            {
              "page_path": event.urlAfterRedirects,
              anonymize_ip: true,
            },
          );
        }
      }),
    ).subscribe();
  }

  disableCookie(): void {
    this._storeCookie(false);
    this.close();
  }

  enableCookie(): void {
    this._storeCookie(true);
    this.close();
  }

  private _storeCookie(store: boolean): void {
    GoogleAnalyticsService._setConfigGoogleAnalyticsStoreCookie(store);

    if (!store) {
      this._cookieService.remove("_ga");
      this._cookieService.remove("_gid");
      this._cookieService.remove("_gat");
      this._cookieService.remove("AMP_TOKEN");
    }

    LocalStorageHelper.setItem(LocalStorageEnum.cookieAuthorized, store ? "true" : "false");
  }

  private static _setConfigGoogleAnalyticsStoreCookie(store: boolean): void {
    gtag(GoogleAnalyticsService._KEY_CONFIG, environment.googleAnalyticsId, {
      "client_storage": store ? "storage" : "none",
    });
  }

  open(): void {
    this._store.dispatch(new AppAction.DisplayCookieConsent(true));
  }

  close(): void {
    this._store.dispatch(new AppAction.DisplayCookieConsent(false));
  }
}
