import {
  HttpClient,
  HttpHeaders,
} from "@angular/common/http";
import {Injectable} from "@angular/core";
import {environment} from "@environments/environment";
import {DownloadToken} from "@models";
import {AccessResourceApiEnum} from "@shared/enums/api.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {saveAs} from "file-saver";
import {
  from,
  Observable,
} from "rxjs";
import {
  catchError,
  take,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  HttpStatus,
  isNotNullNorUndefined,
  isNullOrUndefined,
  isTrue,
  NotificationService,
  OAuth2Service,
  urlSeparator,
} from "solidify-frontend";
import streamSaver from "streamsaver";
import {WritableStream} from "web-streams-polyfill/ponyfill";

@Injectable({
  providedIn: "root",
})
export class DownloadService {
  constructor(private oauth2Service: OAuth2Service,
              private httpClient: HttpClient,
              private readonly _apiService: ApiService,
              private readonly _notificationService: NotificationService) {}

  download(isPublic: boolean, url: string, fileName: string, size?: number, archiveId?: string): void {
    this._notificationService.showInformation(LabelTranslateEnum.fileDownloadStart);
    if (isPublic) {
      this.downloadBrowserNatif(url);
    } else {
      if (isNotNullNorUndefined(archiveId) && environment.production) {
        this._getAndStoreDownloadTokenThenDownload(url, archiveId);
      } else {
        this._downloadPrivateWithoutDownloadToken(url, fileName, size);
      }
    }
  }

  downloadBrowserNatif(url: string): void {
    window.open(url, "_self");
  }

  private _downloadPrivateWithoutDownloadToken(url: string, fileName: string, size?: number): void {
    this.downloadWithMitm(url, fileName, size).pipe(
      tap(result => {
        if (result === HttpStatus.OK) {
          this._notificationService.showSuccess(LabelTranslateEnum.fileDownloadedSuccessfully);
        } else if (result === HttpStatus.FORBIDDEN) {
          this._notificationService.showError(LabelTranslateEnum.fileDownloadForbidden);
        } else {
          this._notificationService.showSuccess(LabelTranslateEnum.fileDownloadFail);
        }
      }),
    ).subscribe();
  }

  private _getAndStoreDownloadTokenThenDownload(url: string, archiveId: string): void {
    this.httpClient.get<DownloadToken>(AccessResourceApiEnum.downloadToken + urlSeparator + archiveId, {
      withCredentials: true,
    }).pipe(
      take(1),
      tap(result => {
        this.downloadBrowserNatif(url);
      }),
      catchError(error => {
        this._notificationService.showError(LabelTranslateEnum.fileDownloadForbidden);
        throw error;
      }),
    ).subscribe();
  }

  downloadWithMitm(url: string, fileName: string, size?: number): Observable<number> {
    if (isNullOrUndefined(streamSaver.WritableStream)) {
      streamSaver.WritableStream = WritableStream;
    }

    if (isTrue(environment.useSelfHostedStreamSaverMitm)) {
      // Allow to avoid to see download from https://jimmywarting.github.io/StreamSaver.js
      streamSaver.mitm = window.location.origin + window.location.pathname.replace(/\/[^\/]+$/, "/") + "mitm.html";
    }
    const fileStream = streamSaver.createWriteStream(fileName, {
      size: size,
      writableStrategy: undefined,
      readableStrategy: undefined,
    });

    const requestHeaders: HeadersInit = new Headers();
    requestHeaders.set("Content-Type", "application/json");
    const token = this.oauth2Service.getAccessToken();
    if (!isNullOrUndefined(token)) {
      requestHeaders.set("Authorization", "Bearer " + token);
    }

    const fetchResult = fetch(url, {
      headers: requestHeaders,
    }).then(res => {
      if (res.status !== HttpStatus.OK) {
        return res.status;
      }
      const readableStream = res.body;
      // const readableStream = new ReadableStream(res.body); // TODO To allow optimize version on firefox but half work and break Chrome download
      const writableStream = streamSaver.WritableStream;
      // Optimized way for supported browser (like Chrome)
      if (writableStream && readableStream.pipeTo) {
        return readableStream.pipeTo(fileStream)
          .then(() => HttpStatus.OK);
      }

      // Standard way for other browser (like Firefox)
      const writer = fileStream.getWriter();
      const reader = res.body.getReader();
      const pump = () => reader.read()
        .then(result => {
          if (result.done) {
            writer.close();
            return HttpStatus.OK;
          } else {
            writer.write(result.value).then(pump);
            return HttpStatus.OK;
          }
        });
      return pump();
    });

    return from(fetchResult);
  }

  // Prefer use download api to stream directly on disk
  downloadInMemory(url: string, fileName: string, withSaveAs: boolean = true): Observable<Blob> {
    let headers = new HttpHeaders();
    headers = headers.set("Content-Disposition", "attachment; filename=" + fileName);

    return this.httpClient.get(url, {
      headers,
      responseType: "blob",
    }).pipe(tap((blobContent: Blob) => {
      if (withSaveAs) {
        saveAs(blobContent, fileName);
      }
    }));
  }
}
