import {HttpErrorResponse} from "@angular/common/http";
import {
  Injectable,
  Injector,
} from "@angular/core";
import {AppAction} from "@app/stores/app.action";
import {Store} from "@ngxs/store";
import {ErrorsSkipperService} from "@shared/services/errors-skipper.service";
import {
  ENVIRONMENT,
  ErrorHelper,
  GlobalErrorsHandlerService,
  HttpStatus,
  isInstanceOf,
  isNullOrUndefined,
  NotificationService,
  SolidifyError,
  MARK_AS_TRANSLATABLE,
} from "solidify-frontend";

@Injectable({
  providedIn: "root",
})
export class DlcmErrorsHandlerService extends GlobalErrorsHandlerService {
  // Error handling is important and needs to be loaded first.
  // Because of this we should manually inject the services with Injector.

  constructor(protected injector: Injector) {
    super(injector);
  }

  handleError(error: Error | HttpErrorResponse | SolidifyError<Error> | SolidifyError<HttpErrorResponse>): void {
    const environment = this.injector.get(ENVIRONMENT);
    const errorSkipper = this.injector.get(ErrorsSkipperService);
    const notificationService = this.injector.get(NotificationService);
    const nativeError = isInstanceOf(error, SolidifyError) ? error.nativeError : error as Error;
    if (nativeError === environment.errorToSkipInErrorHandler) {
      return;
    }
    const httpErrorKeyToSkipInErrorHandler = environment.httpErrorKeyToSkipInErrorHandler;

    if (nativeError instanceof HttpErrorResponse) {
      if (errorSkipper.shouldBeSkipped(nativeError.status)) {
        return;
      }
      if (nativeError.status === 0) {
        notificationService.showError(MARK_AS_TRANSLATABLE("error.notification.backendUnavailable"));
      }
      if (ErrorHelper.isErrorToTreat(nativeError, httpErrorKeyToSkipInErrorHandler)) {
        // Server Error
        if (nativeError.status === HttpStatus.UNAUTHORIZED) {
          if (isNullOrUndefined(nativeError.error) || nativeError.error.error === ErrorHelper.INVALID_TOKEN_ERROR) {
            const store = this.injector.get(Store);
            store.dispatch(new AppAction.Logout());
          }
        }
      }
    }

    super.handleError(error);

    console.error(error);
  }
}
