import {Injectable} from "@angular/core";
import {MatDialog} from "@angular/material/dialog";
import {FirstLoginDialog} from "@app/components/dialogs/first-login/first-login.dialog";
import {AppAction} from "@app/stores/app.action";
import {DepositTabStatusEnum} from "@deposit/enums/deposit-tab-status.enum";
import {Navigate} from "@ngxs/router-plugin";
import {Store} from "@ngxs/store";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {
  DepositRoutesEnum,
  RoutesEnum,
} from "@shared/enums/routes.enum";
import {TourRouteIdEnum} from "@shared/enums/tour-route-id.enum";
import {TourEnum} from "@shared/enums/tour.enum";
import {TourService} from "ngx-tour-md-menu";
import {INgxmStepOption} from "ngx-tour-md-menu/lib/step-option.interface";
import {Observable} from "rxjs";
import {
  take,
  tap,
} from "rxjs/operators";
import {
  isNotNullNorUndefined,
  isNullOrUndefined,
  isTrue,
  MARK_AS_TRANSLATABLE,
  urlSeparator,
} from "solidify-frontend";

@Injectable({
  providedIn: "root",
})
export class AppTourService {
  isInTour: boolean;

  defaultOptions: INgxmStepOption = {
    preventScrolling: true,
    endBtnTitle: LabelTranslateEnum.close,
    enableBackdrop: true,
  };

  mainTourSteps: INgxmStepOption[] = [
    {
      anchorId: TourEnum.mainMenuOrder,
      content: MARK_AS_TRANSLATABLE("tour.main.menuArchiveOrders.content"),
    },
    {
      anchorId: TourEnum.mainMenuDeposit,
      content: MARK_AS_TRANSLATABLE("tour.main.menuDeposit.content"),
    },
    {
      anchorId: TourEnum.mainMenuPreservationSpace,
      content: MARK_AS_TRANSLATABLE("tour.main.menuPreservationSpace.content"),
    },
    {
      anchorId: TourEnum.mainSearchBar,
      content: MARK_AS_TRANSLATABLE("tour.main.searchBar.content"),
      route: RoutesEnum.homePage,
    },
  ];

  depositListTourSteps: INgxmStepOption[] = [
    {
      anchorId: TourEnum.commonContent,
      title: LabelTranslateEnum.deposit,
      content: MARK_AS_TRANSLATABLE("tour.deposit.list.intro.content"),
      route: RoutesEnum.deposit + urlSeparator + TourRouteIdEnum.tourOrgUnitId + urlSeparator + DepositTabStatusEnum.inProgress,
    },
    {
      anchorId: TourEnum.depositCreate,
      content: MARK_AS_TRANSLATABLE("tour.deposit.list.create.content"),
    },
    {
      anchorId: TourEnum.depositStatusTabs,
      content: MARK_AS_TRANSLATABLE("tour.deposit.list.statusTabs.content"),
    },
    {
      anchorId: TourEnum.depositOrgUnitSelector,
      content: MARK_AS_TRANSLATABLE("tour.deposit.list.orgunitSelector.content"),
    },
  ];

  depositMetadataTourSteps: INgxmStepOption[] = [
    {
      anchorId: TourEnum.depositDetail,
      content: MARK_AS_TRANSLATABLE("tour.deposit.metadata.intro.content"),
      route: RoutesEnum.deposit + urlSeparator + TourRouteIdEnum.tourOrgUnitId + urlSeparator + DepositRoutesEnum.detail + urlSeparator + TourRouteIdEnum.tourDepositId + urlSeparator + DepositRoutesEnum.metadata,
    },
    {
      anchorId: TourEnum.depositMetadataThumbnail,
      content: MARK_AS_TRANSLATABLE("tour.deposit.metadata.thumbnail.content"),
    },
  ];

  depositDataTourSteps: INgxmStepOption[] = [
    {
      anchorId: TourEnum.depositDataUpload,
      content: MARK_AS_TRANSLATABLE("tour.deposit.data.upload.content"),
      route: RoutesEnum.deposit + urlSeparator + TourRouteIdEnum.tourOrgUnitId + urlSeparator + DepositRoutesEnum.detail + urlSeparator + TourRouteIdEnum.tourDepositId + urlSeparator + DepositRoutesEnum.data,
    },
    {
      anchorId: TourEnum.depositDetail,
      content: MARK_AS_TRANSLATABLE("tour.deposit.data.remindTooltip.content"),
    },
  ];

  preservationSpaceTourSteps: INgxmStepOption[] = [
    {
      anchorId: TourEnum.preservationSpaceTabs,
      title: LabelTranslateEnum.preservationSpace,
      content: MARK_AS_TRANSLATABLE("tour.preservationSpace.tabs.content"),
      route: RoutesEnum.preservationSpace,
    },
  ];

  preservationSpaceOrgUnitTourSteps: INgxmStepOption[] = [
    {
      anchorId: TourEnum.preservationSpaceOrgUnitTab,
      title: LabelTranslateEnum.organizationalUnit,
      content: MARK_AS_TRANSLATABLE("tour.preservationSpace.orgUnit.tab.content"),
      route: RoutesEnum.preservationSpaceOrganizationalUnitDetail + urlSeparator + TourRouteIdEnum.tourOrgUnitId,
    },
    {
      anchorId: TourEnum.preservationSpaceOrgUnitThumbnail,
      content: MARK_AS_TRANSLATABLE("tour.preservationSpace.orgUnit.thumbnail.content"),
    },
    {
      anchorId: TourEnum.preservationSpaceOrgUnitPersonRole,
      content: MARK_AS_TRANSLATABLE("tour.preservationSpace.orgUnit.personRole.content"),
    },
  ];

  homeSearchTourSteps: INgxmStepOption[] = [
    {
      anchorId: TourEnum.commonContent,
      title: LabelTranslateEnum.search,
      content: MARK_AS_TRANSLATABLE("tour.home.intro.content"),
      route: RoutesEnum.homeSearch,
    },
    {
      anchorId: TourEnum.homeSearchSearchBar,
      content: MARK_AS_TRANSLATABLE("tour.home.searchBar.content"),
    },
    {
      anchorId: TourEnum.homeSearchListTiles,
      content: MARK_AS_TRANSLATABLE("tour.home.listTiles.content"),
    },
    {
      anchorId: TourEnum.homeSearchFacets,
      content: MARK_AS_TRANSLATABLE("tour.home.facets.content"),
    },
  ];

  constructor(private readonly _store: Store,
              private readonly _matDialog: MatDialog,
              private readonly _tourService: TourService) {
  }

  runFullTour(): void {
    setTimeout(() => {
      this._matDialog.open(FirstLoginDialog, {
        maxWidth: "400px",
      }).afterClosed().pipe(
        take(1),
        tap(result => {
          if (isTrue(result)) {
            const steps = [
              ...this.mainTourSteps,
              ...this.homeSearchTourSteps,
              // ...this.depositListTourSteps,
              // ...this.depositMetadataTourSteps,
              // ...this.depositDataTourSteps,
              // ...this.preservationSpaceTourSteps,
              // ...this.preservationSpaceOrgUnitTourSteps,
            ];
            this._initializedTour(steps, {});
            // TODO FOR TRAVEL ON ALL STEPS, MANAGE CASE END TO REDIRECT TO HOME AND REMOVE IS TOUR MODE FROM STATE. PROBLEM, END SI CALLED WHEN LAZY LOADING
            this._endTourObservable(RoutesEnum.homePage).subscribe();
            this._startTour();
          }
        }),
      ).subscribe();
    }, 1000);
  }

  runTour(steps: INgxmStepOption[], extraOptions: INgxmStepOption | undefined = undefined, navigationEnd: string | undefined = undefined): void {
    this._initializedTour(steps, isNullOrUndefined(extraOptions) ? {} : extraOptions);
    this._endTourObservable(navigationEnd).subscribe();
    this._startTour();
  }

  private _initializedTour(steps: INgxmStepOption[], extraOptions: INgxmStepOption = {}): void {
    this._tourService.initialize(steps, {...this.defaultOptions, ...extraOptions});
  }

  private _endTourObservable(navigationEnd: string | undefined): Observable<void> {
    return this._tourService.end$.asObservable().pipe(
      take(1),
      tap(() => {
        this._store.dispatch(new AppAction.SetIsInTourMode(false));
        if (isNotNullNorUndefined(navigationEnd)) {
          this._store.dispatch(new Navigate([navigationEnd]));
        }
      }),
    );
  }

  private _startTour(): void {
    this._store.dispatch(new AppAction.SetIsInTourMode(true));
    this._tourService.start();
  }
}
