import {Injectable} from "@angular/core";
import {
  SwUpdate,
  UpdateActivatedEvent,
  UpdateAvailableEvent,
} from "@angular/service-worker";
import {
  BehaviorSubject,
  from,
  fromEvent,
  merge,
  Observable,
  of,
  timer,
} from "rxjs";
import {
  mapTo,
  skipUntil,
} from "rxjs/operators";
import {LoggingService} from "solidify-frontend";

@Injectable({
  providedIn: "root",
})
export class AppStatusService {
  swUpdates: BehaviorSubject<string | undefined> = new BehaviorSubject(undefined);

  constructor(private updates: SwUpdate,
              private loggingService: LoggingService) {
    if (this.updates.isEnabled) {
      timer(5000, 30000).subscribe(() => {
        this.updates.checkForUpdate();
      });

      // Notify when new service worker version is available
      this.updates.available.subscribe((event: UpdateAvailableEvent) => {
        this.loggingService.logInfo("[AppUpdateAvailable] current version: ", event.current);
        this.loggingService.logInfo("[AppUpdateAvailable] available version: ", event.available);

        if (event.available) {
          this.swUpdates.next(event.available.hash.slice(0, 4));
        }
      });

      // Notify when service worker start serving a new version
      this.updates.activated.subscribe((event: UpdateActivatedEvent) => {
        this.loggingService.logInfo("[AppUpdateActivated] old version was: ", event.previous);
        this.loggingService.logInfo("[AppUpdateActivated] new version is: ", event.current);
      });
    }
  }

  reloadApp(): Observable<void> {
    return from(this.updates.activateUpdate().then(() => document.location.reload()));
  }

  get updateAvailable(): Observable<string> {
    return this.swUpdates.asObservable();
  }

  get online(): Observable<boolean> {
    const windowsEvent: Observable<boolean> = merge(
      fromEvent(window, "online").pipe(mapTo(true)),
      fromEvent(window, "offline").pipe(mapTo(false)),
    );

    return merge(
      of(navigator.onLine),
      windowsEvent.pipe(skipUntil(timer(3000))), // Allow to prevent "online event" when we arrived in app in offline mode.
    );
    // TOFIX : Find a way to know when back to online in the case we arrived in app in offline mode.
    // In this case fromEvent(window, "online") dispatch an event, but should not. Then when we go back in online mode we don't get event online...
  }
}
